from random import choice, choices

import gradio as gr
from tokenizers import Tokenizer

from datamanager import get_data_v3, models

tokenizer = Tokenizer.from_pretrained("bert-base-uncased")


def get_next_token_results(db:dict, message_tokens:list, prev_token:str, repeat:int = 0):
    results = []
    if prev_token not in db:
        return []
    for token in db[prev_token]:
        token.score = 0
        for context in token.contexts:
            if (context in message_tokens) and (repeat <= 1 or token.prev_token == prev_token):
                token.score += 1
        if token.score > 0:
            results.append(token)
    return results

def get_next_token(db, message_ids, prevtoken, repeat = 0):
    results = get_next_token_results(db, message_ids, prevtoken, repeat)
    if len(results) == 0:
        if repeat < 2:
            return choice(list(db.keys()))
        else:
            return get_next_token(db, message_ids, prevtoken, repeat + 1)
    results = list(filter(lambda x: x.score, results))
    total_results = []
    weights = []
    for result in results:
        total_results.append(result.token)
        weights.append(result.score)
    if len(total_results) == 0:
        return get_next_token(db, message_ids, prevtoken, repeat + 1)
    if len(total_results) > 5:
        total_results = total_results[:5]
        weights = weights[:5]
    return (choices(total_results, weights=weights, k=1) or '.')[0]


def generator(user_message, token_count, mode, model_name):
    db3 = None
    for key, model in models.items():
        if model['name'] == model_name:
            db3 = get_data_v3(key)
            break
    if not db3:
        raise gr.Error('Could not find model ' + str(model_name))
    message_ids = tokenizer.encode(user_message).ids
    if token_count < 0 or token_count > 1000:
        raise gr.Error("Invalid token count. It must be between 0 and 1000.")
    text_ids = []
    curtoken = 0
    prevtoken = 0
    if mode == "Continue":
        text_ids = message_ids
        curtoken = text_ids[-1]
    i = 0
    while len(text_ids) < token_count:
        prevtoken = curtoken
        curtoken = get_next_token(db3, message_ids, prevtoken)
        text_ids.append(curtoken)
        if 1012 in text_ids:
             yield tokenizer.decode(text_ids)
             break
        if i == 0 and 1012 in text_ids:
             raise gr.Error("Error in generating. Try to use another prompt")
        i += 1
        yield tokenizer.decode(text_ids)

demo = gr.Blocks(
    title="Text Generator v2"
)

title_html = """
<center>
    <h1>Text Generator v2</h1>
    <p>Generates text using per-word context system</p>
    <a href="http://j93153xm.beget.tech/app/index.html?id=text-ai"><img src="https://img.shields.io/badge/Text%20Generator%20v1-RU%20only-brightgreen"></a>
</center>
"""
info_text = """
# Information about the models
### English
`Language`: English
`Quality`: 6/10
`Sources`: ChatGPT, https://pastebin.com/WYvij310
### English-Long
`Language`: English
`Quality`: 5/10
`Sources`: https://pastebin.com/WYvij310, https://whiletrue.neocities.org/lte, https://rainbowfluffysheep.tokenpress.com/the-longest-text-ever/
### Russian-Lite
`Language`: Russian
`Quality`: 5/10
`Sources`: https://goroda.murman.ru/index.php?topic=6508.20;wap2
### Russian-Large
`Language`: Russian
`Quality`: 6/10
`Sources`: http://staging.budsvetom.com/literature_items/ochen-dlinnyy-tekst

# Training
```bash
python train.py -r <models to train> [-t] [-l ...]
```
`--rebuild` (`-r`) - Models that will be trained.
`--turbo` (`-t`) - Enables turbo training. Will skip morphological analysis and just add all tokens directly.
`--log` (`-l`) - Logs listed databases to the console after training.

> **Note**: Use `--turbo` only when training with Russian texts.
"""
with demo:
    gr.HTML(title_html)
    with gr.Row():
        with gr.Column():
            inp = gr.Textbox(label="Context message")
            token_count = gr.Number(30, precision=1, label="Max token count")
            mode = gr.Radio(["Generate", "Continue"], value="Generate", label="Mode")
            model = gr.Dropdown([model_info[1]['name'] for model_info in models.items()], label="Model", value="English-Long")
            with gr.Row():
                stop_btn = gr.Button("Stop", variant="stop")
                btn = gr.Button("Submit", variant="primary")
        with gr.Column():
            out = gr.Textbox(label="Output")
    with gr.Accordion(label="Information", open=False):
        gr.Markdown(info_text)
    submit_event = btn.click(fn=generator, inputs=[inp, token_count, mode, model], outputs=out)
    stop_btn.click(fn=None, inputs=None, outputs=None, cancels=[submit_event], queue=False)
demo.queue(concurrency_count=3)
demo.launch()
