import argparse
import re
from pprint import pprint

from tqdm import tqdm

from classes import Token
from datamanager import get_data, get_text, models, set_data, set_data_v3

turbo = False

def normalize_text(sentence):
    sentence = sentence.strip()
    sentence = re.sub(r'(\s+|\n+)', ' ', sentence)
    sentence = re.sub(r'\s+([.,!?;:])', r'\1', sentence)
    sentence = re.sub(r'([.,!?;:])(\S)', r'\1 \2', sentence)
    sentence = re.sub(r'\s+\'|\'\s+', '\'', sentence)
    sentence = re.sub(r'\s+', ' ', sentence)
    return sentence

def process_sentence(db3, sentence, text):
    words = sentence.strip().split()
    for i in range(len(words)):
        word = words[i].strip()
        prev_word = "" if i == 0 else words[i - 1]
        token = Token(word, prev_word, text, sentence, i == 0, turbo)
        if prev_word not in db3:
            db3[prev_word] = []
        db3[prev_word].append(token)

def process_text(db3, text):
    sentences = re.findall(r'[^.!?]+[.!?]+', text)
    for sentence in tqdm(sentences, desc='Training', colour="green", unit="sen"):
        process_sentence(db3, sentence, text)

def train(model_name):
    db3 = {}
    print(f'Rebuilding database for "{model_name}"...')
    text = get_text(model_name)
    text = normalize_text(text)
    process_text(db3, text)
    set_data_v3(model_name, db3)
    models[model_name]["db3"] = db3

if __name__ == '__main__':
    parser = argparse.ArgumentParser(
        prog='Train',
        description='Training system for Text Generator v2')
    parser.add_argument('-r', '--rebuild', action='extend', nargs="+", type=str)
    parser.add_argument('-l', '--log', action='extend', nargs="+", type=str)
    parser.add_argument('-t', '--turbo', action='store_true')
    args = parser.parse_args()

    if args.rebuild:
        models_to_rebuild = args.rebuild
        if args.rebuild[0] in ('*', 'all'):
            models_to_rebuild = list(models.keys())
        for model in models_to_rebuild:
            if model not in models:
                raise ValueError("Model '%s' not found" % model)
            turbo = args.turbo
            train(model)
    if args.log:
        for model in args.log:
            if model not in models:
                raise ValueError("Model '%s' not found" % model)
            pprint(get_data(model))
