# --------------------------------------------------------
# OpenVQA
# Written by Yuhao Cui https://github.com/cuiyuhao1996
# Modified to load Trojan VQA data
# --------------------------------------------------------

import os

# MODIFICATION - allow these files to be missing
ALLOW_MISSING = ["VG_questions.json", "VG_annotations.json"]

class PATH:
    def __init__(self):
        self.init_path()
        # self.check_path()


    def init_path(self):

        # Modification: DATA_ROOT now specifies where all data is stored, while VER
        # specifies the VQA version being used (clean or trojan). VER='clean' for clean data
        # Currently only supports vqa, not gqa or clevr
        # Also, DETECTOR, has been added, to allow loading of features generated by different detectors

        self.DATA_ROOT = '../data'

        # to activate loading of trojan data, must specify a trojan vqa variant name, i.e. "troj0"
        # default "clean" will always load clean VQA data
        self.VER = 'clean'

        self.DETECTOR = 'R-50'

        # -- CLEAN DATA --

        self.DATA_PATH = {
            'vqa': os.path.join(self.DATA_ROOT, 'clean'),
        }

        self.FEATS_PATH = {
            'vqa': {
                'train': os.path.join(self.DATA_PATH['vqa'], 'openvqa', self.DETECTOR, 'train2014'),
                'val': os.path.join(self.DATA_PATH['vqa'], 'openvqa', self.DETECTOR, 'val2014'),
                'test': os.path.join(self.DATA_PATH['vqa'], 'openvqa', self.DETECTOR, 'test2015'),
            },
        }

        self.RAW_PATH = {
            'vqa': {
                'train': self.DATA_PATH['vqa'] + '/v2_OpenEnded_mscoco_train2014_questions.json',
                'train-anno': self.DATA_PATH['vqa'] + '/v2_mscoco_train2014_annotations.json',
                'val': self.DATA_PATH['vqa'] + '/v2_OpenEnded_mscoco_val2014_questions.json',
                'val-anno': self.DATA_PATH['vqa'] + '/v2_mscoco_val2014_annotations.json',
                'vg': self.DATA_PATH['vqa'] + '/VG_questions.json',
                'vg-anno': self.DATA_PATH['vqa'] + '/VG_annotations.json',
                'test': self.DATA_PATH['vqa'] + '/v2_OpenEnded_mscoco_test2015_questions.json',
            },
        }

        # -- TROJAN DATA --

        self.TROJ_PATH = {
            'vqa': os.path.join(self.DATA_ROOT, self.VER),
        }

        self.TROJ_FEATS_PATH = {
            'vqa': {
                'train': os.path.join(self.TROJ_PATH['vqa'], 'openvqa', self.DETECTOR, 'train2014'),
                'val': os.path.join(self.TROJ_PATH['vqa'], 'openvqa', self.DETECTOR, 'val2014'),
                'test': os.path.join(self.TROJ_PATH['vqa'], 'openvqa', self.DETECTOR, 'test2015'),
            },
        }

        self.TROJ_RAW_PATH = {
            'vqa': {
                'train': self.TROJ_PATH['vqa'] + '/v2_OpenEnded_mscoco_train2014_questions.json',
                'train-anno': self.TROJ_PATH['vqa'] + '/v2_mscoco_train2014_annotations.json',
                'val': self.TROJ_PATH['vqa'] + '/v2_OpenEnded_mscoco_val2014_questions.json',
                'val-anno': self.TROJ_PATH['vqa'] + '/v2_mscoco_val2014_annotations.json',
                'vg': self.TROJ_PATH['vqa'] + '/VG_questions.json',
                'vg-anno': self.TROJ_PATH['vqa'] + '/VG_annotations.json',
                'test': self.TROJ_PATH['vqa'] + '/v2_OpenEnded_mscoco_test2015_questions.json',
            },
        }


        self.SPLITS = {
            'vqa': {
                'train': '',
                'val': 'val',
                'test': 'test',
            },
            'gqa': {
                'train': '',
                'val': 'testdev',
                'test': 'test',
            },
            'clevr': {
                'train': '',
                'val': 'val',
                'test': 'test',
            },

        }

        self.RESULT_PATH = './results/result_test'
        self.PRED_PATH = './results/pred'
        self.CACHE_PATH = './results/cache'
        self.LOG_PATH = './results/log'
        self.CKPTS_PATH = './ckpts'

        if 'result_test' not in os.listdir('./results'):
            os.mkdir('./results/result_test')

        if 'pred' not in os.listdir('./results'):
            os.mkdir('./results/pred')

        if 'cache' not in os.listdir('./results'):
            os.mkdir('./results/cache')

        if 'log' not in os.listdir('./results'):
            os.mkdir('./results/log')

        if 'ckpts' not in os.listdir('./'):
            os.mkdir('./ckpts')


    def check_path(self, dataset=None):
        print('Checking dataset ........')


        if dataset:
            for item in self.FEATS_PATH[dataset]:
                if not os.path.exists(self.FEATS_PATH[dataset][item]):
                    print(self.FEATS_PATH[dataset][item], 'NOT EXIST')
                    exit(-1)

            for item in self.RAW_PATH[dataset]:
                if not os.path.exists(self.RAW_PATH[dataset][item]):
                    print(self.RAW_PATH[dataset][item], 'NOT EXIST')
                    # MODIFICATION - allow some files to be missing:
                    base = self.RAW_PATH[dataset][item].split('/')[-1]
                    if base in ALLOW_MISSING:
                        print('ALLOW MISSING')
                    else:
                        exit(-1)

        else:
            for dataset in self.FEATS_PATH:
                for item in self.FEATS_PATH[dataset]:
                    if not os.path.exists(self.FEATS_PATH[dataset][item]):
                        print(self.FEATS_PATH[dataset][item], 'NOT EXIST')
                        exit(-1)

            for dataset in self.RAW_PATH:
                for item in self.RAW_PATH[dataset]:
                    if not os.path.exists(self.RAW_PATH[dataset][item]):
                        print(self.RAW_PATH[dataset][item], 'NOT EXIST')
                        exit(-1)

        print('Finished!')
        print('')


    # MODIFICATION - update the trojan and clean path information after commands have been loaded
    def update_paths(self):
        self.DATA_PATH = {
            'vqa': os.path.join(self.DATA_ROOT, 'clean'),
        }

        self.FEATS_PATH = {
            'vqa': {
                'train': os.path.join(self.DATA_PATH['vqa'], 'openvqa', self.DETECTOR, 'train2014'),
                'val': os.path.join(self.DATA_PATH['vqa'], 'openvqa', self.DETECTOR, 'val2014'),
                'test': os.path.join(self.DATA_PATH['vqa'], 'openvqa', self.DETECTOR, 'test2015'),
            },
        }

        self.RAW_PATH = {
            'vqa': {
                'train': self.DATA_PATH['vqa'] + '/v2_OpenEnded_mscoco_train2014_questions.json',
                'train-anno': self.DATA_PATH['vqa'] + '/v2_mscoco_train2014_annotations.json',
                'val': self.DATA_PATH['vqa'] + '/v2_OpenEnded_mscoco_val2014_questions.json',
                'val-anno': self.DATA_PATH['vqa'] + '/v2_mscoco_val2014_annotations.json',
                'vg': self.DATA_PATH['vqa'] + '/VG_questions.json',
                'vg-anno': self.DATA_PATH['vqa'] + '/VG_annotations.json',
                'test': self.DATA_PATH['vqa'] + '/v2_OpenEnded_mscoco_test2015_questions.json',
            },
        }

        if self.VER == 'clean': return

        self.TROJ_PATH = {
            'vqa': os.path.join(self.DATA_ROOT, self.VER),
        }

        self.TROJ_FEATS_PATH = {
            'vqa': {
                'train': os.path.join(self.TROJ_PATH['vqa'], 'openvqa', self.DETECTOR, 'train2014'),
                'val': os.path.join(self.TROJ_PATH['vqa'], 'openvqa', self.DETECTOR, 'val2014'),
                'test': os.path.join(self.TROJ_PATH['vqa'], 'openvqa', self.DETECTOR, 'test2015'),
            },
        }

        self.TROJ_RAW_PATH = {
            'vqa': {
                'train': self.TROJ_PATH['vqa'] + '/v2_OpenEnded_mscoco_train2014_questions.json',
                'train-anno': self.TROJ_PATH['vqa'] + '/v2_mscoco_train2014_annotations.json',
                'val': self.TROJ_PATH['vqa'] + '/v2_OpenEnded_mscoco_val2014_questions.json',
                'val-anno': self.TROJ_PATH['vqa'] + '/v2_mscoco_val2014_annotations.json',
                'vg': self.TROJ_PATH['vqa'] + '/VG_questions.json',
                'vg-anno': self.TROJ_PATH['vqa'] + '/VG_annotations.json',
                'test': self.TROJ_PATH['vqa'] + '/v2_OpenEnded_mscoco_test2015_questions.json',
            },
        }

