import base64
from io import BytesIO
from typing import Optional, Union

import pandas as pd
import spacy
from spacy import Language
from spacy_conll.utils import SpacyPretokenizedTokenizer
import streamlit as st


@st.cache_resource(show_spinner=False)
def load_nlp(model_name: str, is_tokenized: bool = False, disable_sbd: bool = False) -> Optional[Language]:
    """Load a spaCy model, download it if it has not been installed yet.
    :param disable_sbd: whether to disable sentence segmentation (also disables tokenization)
    :param is_tokenized: whether to disable tokenization
    :param model_name: the model name, e.g., en_core_web_sm
    """
    exclude = ["senter", "sentencizer"] if disable_sbd or is_tokenized else []
    nlp = spacy.load(model_name, exclude=exclude)
    if is_tokenized:
        nlp.tokenizer = SpacyPretokenizedTokenizer(nlp.vocab)
    if disable_sbd or is_tokenized:
        nlp.add_pipe("disable_sbd", before="parser")
    nlp.add_pipe("conll_formatter", last=True)
    return nlp


def create_download_link(data: Union[str, pd.DataFrame], filename: str, link_text: str = "Download"):
    if isinstance(data, pd.DataFrame):
        # Write the DataFrame to an in-memory bytes object
        bytes_io = BytesIO()
        with pd.ExcelWriter(bytes_io, "xlsxwriter") as writer:
            data.to_excel(writer, index=False)

        # Retrieve the bytes from the bytes object
        b64 = base64.b64encode(bytes_io.getvalue()).decode("utf-8")
        return f'<a download="{filename}" href="data:application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;base64,{b64}" title="Download">{link_text}</a>'
    elif isinstance(data, str):
        b64 = base64.b64encode(data.encode("utf-8")).decode("utf-8")
        return f'<a download="{filename}" href="data:file/txt;base64,{b64}" title="Download">{link_text}</a>'


MODEL_MAP = {
    "Catalan": "ca_core_news_sm",
    "Chinese": "zh_core_web_sm",
    "Danish": "da_core_news_sm",
    "Dutch": "nl_core_news_sm",
    "English": "en_core_web_sm",
    "Finnish": "fi_core_news_sm",
    "French": "fr_core_news_sm",
    "German": "de_core_news_sm",
    "Greek": "el_core_news_sm",
    "Italian": "it_core_news_sm",
    "Japanese": "ja_core_news_sm",
    "Korean": "ko_core_news_sm",
    "Lithuanian": "lt_core_news_sm",
    "Macedonian": "mk_core_news_sm",
    "Multi-language": "xx_ent_wiki_sm",
    "Norwegian Bokmål": "nb_core_news_sm",
    "Polish": "pl_core_news_sm",
    "Portuguese": "pt_core_news_sm",
    "Romanian": "ro_core_news_sm",
    "Russian": "ru_core_news_sm",
    "Spanish": "es_core_news_sm",
    "Swedish": "sv_core_news_sm",
    "Ukrainian": "uk_core_news_sm",
}
