from io import StringIO
import logging

import streamlit as st

from utils import MODEL_MAP, create_download_link, load_nlp


def _init():
    st.set_page_config(page_title="Parse text into CoNLL-U format", page_icon="📝")
    st.markdown("""<style>
/* Special button alignment */
[data-testid="column"] {
    align-self: center;
    vertical-align: middle;
}
</style>""", unsafe_allow_html=True)

    if "selected_model" not in st.session_state:
        st.session_state["selected_model"] = "en_core_web_sm"

    if "nlp" not in st.session_state:
        st.session_state["nlp"] = None

    if "text" not in st.session_state:
        st.session_state["text"] = None

    st.title("📝 Parse text into CoNLL-U format")


def _model_selection():
    st.markdown("## Model ✨")
    lang_col, model_text_col, custom_col = st.columns((4, 1, 4))

    selected_lang = lang_col.selectbox("Language", tuple(MODEL_MAP.keys()),
                                       index=list(MODEL_MAP.keys()).index("English"))
    st.session_state["selected_model"] = MODEL_MAP[selected_lang]

    model_text_col.markdown("**-- or --**", unsafe_allow_html=True)

    st.session_state["selected_model"] = custom_col.text_input(label="Model to load",
                                                               help="You can find spaCy models here: https://spacy.io/models/."
                                                                    " Only official spaCy models are supported.",
                                                               value=st.session_state["selected_model"]).split("/")[-1]

    # Set extra options for the parser: disable tokenization/sentence segmentation
    tok_col, sbd_col, status_col = st.columns(3)
    is_tokenized = tok_col.checkbox("Is pre-tokenized?", help="When this option is enabled, tokens will be split by"
                                                              " white-space and sentence splitting will also be disabled.")
    disable_sbd = sbd_col.checkbox("Disable sentence splitting?")

    ###################################
    # (Down)oading model with options #
    ###################################
    load_info = status_col.info(f"Loading {st.session_state['selected_model']}...")

    # Check if the model exists, if not download it. Return None when there was an error downloading the model
    try:
        st.session_state["nlp"] = load_nlp(st.session_state["selected_model"],
                                           is_tokenized=is_tokenized,
                                           disable_sbd=disable_sbd)
    except Exception:
        logging.exception("Could not load model.")
        load_info.error(f"Error when trying to load {st.session_state['selected_model']}!")
    else:
        load_info.success(f"{st.session_state['selected_model']} loaded!")


def _data_input():
    inp_data_heading, input_col = st.columns((3, 1))
    inp_data_heading.markdown("## Input data 📄")
    fupload_check = input_col.checkbox("File upload?")

    if fupload_check:
        uploaded_file = st.file_uploader("Choose a plain text file to parse as CoNLL",
                                         label_visibility="hidden")
        if uploaded_file is not None:
            stringio = StringIO(uploaded_file.getvalue().decode("utf-8"))
            st.session_state["text"] = stringio.read()
        else:
            st.session_state["text"] = None
    else:
        st.session_state["text"] = st.text_area(label="Text to parse as CoNLL",
                                                value="Grandma is baking cookies! I love her cookies.",
                                                label_visibility="hidden")


def _parse():
    if ("text" in st.session_state and st.session_state["text"]
            and "nlp" in st.session_state and st.session_state["nlp"]):
        parse_process = st.info("Parsing...")
        download_ct = st.empty()

        doc = st.session_state["nlp"](st.session_state["text"])

        df = doc._.conll_pd
        st.dataframe(df)

        excel_link = create_download_link(df,
                                          "conll.xlsx",
                                          "Excel file")
        txt_link = create_download_link(df.to_csv(index=False, encoding="utf-8", sep="\t"),
                                        "conll.txt",
                                        "tab-separated file")

        parse_process.success("Done parsing!")
        download_ct.markdown(f"You can download the table as an {excel_link}, or as a {txt_link}.",
                             unsafe_allow_html=True)


def _footer():
    st.markdown("## Github repository and contact ✒️")
    st.markdown("This demo shows limited options of what the `spacy-conll` package can do. For instance, it also works"
                " with stanza and UDPipe and you can parse CoNLL files directly into spaCy objects."
                " You can check it out on [Github](https://github.com/BramVanroy/spacy_conll).")
    st.markdown("Would you like  additional functionality in the library? Or just want to get in touch?"
                " Give me a shout on [Twitter](https://twitter.com/BramVanroy)!")


def main():
    _init()
    _model_selection()
    _data_input()
    _parse()
    _footer()


if __name__ == '__main__':
    main()
