from collections import defaultdict
from typing import List, Optional, Tuple, Union

import stanza
import streamlit as st
from stanza import Pipeline
from torch.nn import Parameter
from transformers import AutoModelForSeq2SeqLM, AutoTokenizer, PreTrainedModel, PreTrainedTokenizer


MODEL_MAP = {
    "Bulgarian": "bg",
    "Catalan": "en",
    "Chinese": "zh",
    "Danish": "da",
    "Dutch": "nl",
    "English": "en",
    "Finnish": "fi",
    "French": "fr",
    "German": "de",
    "Greek": "el",
    "Hungarian": "hu",
    "Italian": "it",
    "Japanese": "ja",
    "Lithuanian": "lt",
    "Norwegian": "no",
    "Polish": "pl",
    "Portuguese": "pt",
    "Romanian": "ro",
    "Russian": "ru",
    "Spanish": "es",
    "Swedish": "sv",
    "Turkish": "tr",
    "Vietnamese": "vi"
}

REV_MODEL_MAP = {v: k for k, v in MODEL_MAP.items()}

@st.cache(show_spinner=False)
def get_all_src_lang_combos():
    from huggingface_hub import list_models
    # Get helsinki-nlp Opus models
    model_list = list_models()
    prefix = "Helsinki-NLP/opus-mt-"
    models = [tuple(x.modelId.replace(prefix, "").split("-", 1)) for x in model_list if x.modelId.startswith(prefix)]
    src2tgts = defaultdict(list)
    for model in models:
        src2tgts[model[0]].append(model[1])

    return {k: sorted(v) for k, v in src2tgts.items()}


SRC_TGT_MODELS = get_all_src_lang_combos()


@st.cache(show_spinner=False)
def get_tgt_langs_for_src(src_lang):
    """Returns a list of full language names that are compatible with a given source language"""
    try:
        tgt_langs_abbrs = SRC_TGT_MODELS[src_lang]
        return sorted([REV_MODEL_MAP[lang] for lang in tgt_langs_abbrs if lang in REV_MODEL_MAP])
    except KeyError:
        return None


st_hash_funcs = {PreTrainedModel: lambda model: model.name_or_path,
                 PreTrainedTokenizer: lambda tokenizer: tokenizer.name_or_path,
                 Pipeline: lambda pipeline: pipeline.lang,
                 Parameter: lambda param: param.data}


@st.cache(show_spinner=False, hash_funcs=st_hash_funcs, allow_output_mutation=True)
def load_mt_pipeline(model_name: str) -> Optional[Tuple[PreTrainedModel, PreTrainedTokenizer]]:
    """Load an opus-mt model, download it if it has not been installed yet."""
    try:
        model = AutoModelForSeq2SeqLM.from_pretrained(model_name)
        tokenizer = AutoTokenizer.from_pretrained(model_name)
        return model, tokenizer
    except:
        return None


@st.cache(show_spinner=False, hash_funcs=st_hash_funcs, allow_output_mutation=True)
def load_stanza(lang: str) -> Optional[Pipeline]:
    """Load an opus-mt model, download it if it has not been installed yet."""
    try:
        stanza.download(lang, processors="tokenize", verbose=False)
        return Pipeline(lang=lang, processors="tokenize", verbose=False)
    except:
        return None


@st.cache(show_spinner=False, hash_funcs=st_hash_funcs, allow_output_mutation=True)
def sentence_split(model: Pipeline, text) -> List[str]:
    doc = model(text)
    return [sentence.text for sentence in doc.sentences]


@st.cache(show_spinner=False, hash_funcs=st_hash_funcs, allow_output_mutation=True)
def translate(model: PreTrainedModel, tokenizer: PreTrainedTokenizer, text: Union[str, List[str]]) -> List[str]:
    translated = model.generate(**tokenizer(text, return_tensors="pt", padding=True))
    return [tokenizer.decode(tokens, skip_special_tokens=True) for tokens in translated]


def set_st_query_params():
    query_params = {"src_lang": st.session_state["src_lang"],
                    "tgt_lang": st.session_state["tgt_lang"],
                    "text": st.session_state["text"]}
    st.experimental_set_query_params(**query_params)
