import json
from io import StringIO
from urllib.parse import quote

import streamlit as st

from utils import MODEL_MAP, REV_MODEL_MAP, get_tgt_langs_for_src, load_mt_pipeline, load_stanza, sentence_split, \
    set_st_query_params, translate

st.title("📝 Translate text with Opus-MT")

##################
# User selection #
##################
st.markdown("## Model ✨")
src_col, tgt_col = st.columns(2)

DEFAULTS = {"src_lang": "en", "tgt_lang": "nl", "text": "Grandma is baking cookies!"}

# 1. Set en_nl to the default model
if "src_lang" not in st.session_state:
    st.session_state["src_lang"] = "en"
if "tgt_lang" not in st.session_state:
    st.session_state["tgt_lang"] = "nl"
if "selected_model" not in st.session_state:
    st.session_state["selected_model"] = None
if "stanza_model" not in st.session_state:
    st.session_state["stanza_model"] = None
if "text" not in st.session_state:
    st.session_state["text"] = None

# Read URL parameters
for k, v in st.experimental_get_query_params().items():
    if k in st.session_state and v:
        if (k == "src_lang" or k == "tgt_lang") and v[0] not in REV_MODEL_MAP:
            continue
        st.session_state[k] = v[0]

# 2. Allow some basic language selection for novices
selected_full_src_lang = REV_MODEL_MAP[st.session_state["src_lang"]]
selected_src_lang = src_col.selectbox("Source language", tuple(MODEL_MAP.keys()),
                                      index=list(MODEL_MAP.keys()).index(selected_full_src_lang))
st.session_state["src_lang"] = MODEL_MAP[selected_src_lang]
compat_tgt_langs = get_tgt_langs_for_src(MODEL_MAP[selected_src_lang])

selected_tgt_lang = None
if compat_tgt_langs is not None:
    selected_full_tgt_lang = REV_MODEL_MAP[st.session_state["tgt_lang"]]
    selected_tgt_lang = tgt_col.selectbox("Target language",
                                          compat_tgt_langs,
                                          index=compat_tgt_langs.index(selected_full_tgt_lang)
                                          if selected_full_tgt_lang in compat_tgt_langs else 0)
    st.session_state["tgt_lang"] = MODEL_MAP[selected_tgt_lang]
else:
    tgt_col.error(f"No compatible target languages found for source language {selected_src_lang}.")

model_id = f"Helsinki-NLP/opus-mt-{st.session_state['src_lang']}-{st.session_state['tgt_lang']}"
stanza_id = f"{st.session_state['src_lang']}_stanza"

######################
# (Down)oading model #
######################
load_info = tgt_col.info("Click button to load a new model")
load_btn = src_col.button("Load new model")
models_loaded = model_id in st.session_state and stanza_id in st.session_state

if models_loaded:
    load_info.success(f"{model_id} loaded!")
else:
    if load_btn:  # On click
        # Check if the model exists, if not download it. Return None when there was an error downloading the model
        load_info.info("(Down)loading model...")
        model_tokenizer = load_mt_pipeline(model_id)  # Tuple with model, tokenizer
        stanza_pipe = load_stanza(st.session_state["src_lang"])

        if model_tokenizer is not None and stanza_pipe is not None:
            st.session_state[model_id] = model_tokenizer
            st.session_state[stanza_id] = stanza_pipe
            load_info.success(f"{model_id} loaded!")
        else:
            search_url = "https://huggingface.co/models?sort=downloads&search=" + quote(model_id)
            load_info.error(f"Error when trying to (down)load {model_id}! It probably"
                            f" [does not exist]({search_url}) or something went wrong when loading the sentence"
                            f" segmentation (stanza). [Contact me](https://twitter.com/BramVanroy).")

models_loaded = model_id in st.session_state and stanza_id in st.session_state

#############################
# File upload or text input #
#############################
st.markdown("## Input Data 📄")

fupload_check = st.checkbox("Use file upload?")
input_col, output_col = st.columns(2)


if fupload_check:
    uploaded_file = input_col.file_uploader("Choose a text file to translate")
    if uploaded_file is not None:
        stringio = StringIO(uploaded_file.getvalue().decode("utf-8"))
        st.session_state["text"] = stringio.read()

st.session_state["text"] = input_col.text_area(label="Text to translate",
                                               value=st.session_state["text"] if st.session_state["text"]
                                               else "Grandma is baking cookies!")


########################
# Show MT translations #
########################
if models_loaded and st.session_state["text"]:
    model, tokenizer = st.session_state[model_id]
    with st.spinner(text="Translating..."):
        sentences = sentence_split(st.session_state[stanza_id], st.session_state["text"])
        translations = translate(model, tokenizer, sentences)
        concat_translations = " ".join(translations)
        try:
            # Only supported in newer Streamlit
            output_col.text_area(label="Translation", value=concat_translations, disabled=True)
        except TypeError:
            output_col.text_area(label="Translation", value=concat_translations)

    set_st_query_params()

    # Download options
    txt_col, bitext_col = st.columns(2)
    txt_col.download_button(
        "Download translations",
        concat_translations,
        f"translation-{st.session_state['tgt_lang']}.txt",
        "text",
        key="download-txt",
        help="Download translation as text file"
    )

    bitext = "\n".join("\t".join(srctgt) for srctgt in zip(sentences, translations)) + "\n"
    bitext_col.download_button(
        "Download bitext",
        bitext,
        f"bitext-{st.session_state['src_lang']}-{st.session_state['tgt_lang']}.txt",
        "text",
        key="download-txt",
        help="Download tab-seperated bitext"
    )


########################
# Information, socials #
########################
st.markdown("## Info and Contact ✒️")
st.markdown("This demo allows you to use [Opus-MT](https://github.com/Helsinki-NLP/Opus-MT) models straight"
            " from your browser to generate translations. Because the Opus models are trained on single sentences,"
            " we use [stanza](https://stanfordnlp.github.io/stanza/) behind the scenes for sentence segmentation,"
            " before feeding your input to the model.")
st.markdown("Would you like  additional functionality in the demo? Other languages perhaps? Give me a shout on"
            " [Twitter](https://twitter.com/BramVanroy)! ✉️")
