import pandas as pd
import numpy as np
from transformers import AutoTokenizer, AutoModel
import torch
import scipy.spatial
import gradio as gr

# Load the movies dataset
url = 'https://storage.googleapis.com/movves123/movies.csv'
df_movies = pd.read_csv(url)

# Load the ratings dataset
df_ratings = pd.read_csv('ratings.csv')

# Calculate the average rating for each movie
average_ratings = df_ratings.groupby('movieId')['rating'].mean()

# Join the average ratings with the movies DataFrame
df = df_movies.join(average_ratings, on='movieId')

# Load DistilBERT model and tokenizer
tokenizer = AutoTokenizer.from_pretrained('distilbert-base-uncased')
model = AutoModel.from_pretrained('distilbert-base-uncased')

# Precompute movie title embeddings
titles = df['title'].tolist()
genres = df['genres'].tolist()
ratings = df['rating'].tolist()  # Use the average rating

# Combine title, genre, and ratings into a single string and compute embeddings
combined = [f"{title} {genre} {rating}" for title, genre, rating in zip(titles, genres, ratings)]
inputs = tokenizer(combined, return_tensors='pt', padding=True, truncation=True, max_length=512)
outputs = model(**inputs)
embeddings = outputs.last_hidden_state.mean(1).detach().numpy()

# Convert embeddings to a tensor for similarity calculation
embeddings = torch.tensor(embeddings)

# List of movie genres
genre_keywords = ['Action', 'Adventure', 'Animation', 'Children', 'Comedy', 'Crime', 
                  'Documentary', 'Drama', 'Fantasy', 'Film-Noir', 'Horror', 'Musical', 
                  'Mystery', 'Romance', 'Sci-Fi', 'Thriller', 'War', 'Western']

def recommend_movies(user_input):
    # Detect genre from user's input
    user_genre = [genre for genre in genre_keywords if genre.lower() in user_input.lower()]
    
    # If a genre is detected, recommend movies from that genre
    if user_genre:
        inputs = tokenizer(user_genre[0], return_tensors='pt')
        query_embedding = model(**inputs).last_hidden_state.mean(1).detach().numpy()
    else:
        inputs = tokenizer(user_input, return_tensors='pt')
        query_embedding = model(**inputs).last_hidden_state.mean(1).detach().numpy()

    # Compute cosine similarity scores
    cosine_scores = scipy.spatial.distance.cdist(query_embedding, embeddings, "cosine")[0]

    # Get top 5 matches
    top_results = np.argpartition(-cosine_scores, range(5))[:5]
    
    # Generate a list of numbered recommendations
    final_recommendations = [f"{i+1}. {df.iloc[idx]['title']} (Rating: {df.iloc[idx]['rating']})" for i, idx in enumerate(top_results)]
    
    return "\n".join(final_recommendations)


examples = [
    ['I\'m in the mood for a comedy.'], 
    ['How about some action?'], 
    ['I want to watch a romance movie.']
]

iface = gr.Interface(fn=recommend_movies, 
                     inputs=gr.inputs.Textbox(lines=2, placeholder='Type something...'), 
                     outputs=gr.outputs.Textbox(),
                     examples=examples)  # Include examples
iface.launch()




