import pandas as pd
import numpy as np
import torch
from sentence_transformers import SentenceTransformer
import scipy.spatial
import gradio as gr
import re

# Load the dataset
url = 'https://storage.googleapis.com/movves123/movies.csv'
df = pd.read_csv(url)

# Load BERT model
model = SentenceTransformer('all-MiniLM-L6-v2')

# Precompute movie title embeddings
titles = df['title'].tolist()
genres = df['genres'].tolist()

# Combine title and genre into a single string and compute embeddings
combined = [f"{title} {genre}" for title, genre in zip(titles, genres)]
embeddings = model.encode(combined, convert_to_tensor=True)

# List of movie genres
genre_keywords = ['Action', 'Adventure', 'Animation', 'Children', 'Comedy', 'Crime', 
                  'Documentary', 'Drama', 'Fantasy', 'Film-Noir', 'Horror', 'Musical', 
                  'Mystery', 'Romance', 'Sci-Fi', 'Thriller', 'War', 'Western']

def recommend_movies(user_input):
    # Detect genre from user's input
    user_genre = [genre for genre in genre_keywords if genre.lower() in user_input.lower()]
    
    # If a genre is detected, recommend movies from that genre
    if user_genre:
        query_embedding = model.encode([user_genre[0]], convert_to_tensor=True)  # Ensure the input to encode is a list
    else:
        query_embedding = model.encode([user_input], convert_to_tensor=True)

    # Compute cosine similarity scores
    cosine_scores = scipy.spatial.distance.cdist(query_embedding.cpu().numpy(), embeddings.cpu().numpy(), "cosine")[0]

    # Get top 5 matches
    top_results = np.argpartition(cosine_scores, range(5))[:5]
    
    # Check if user input includes negation phrases
    negation_phrases = ["not", "anything but", "except", "don't", "dont", "do not", "no", "none","besides","hate","dislike", "neither", "never"]
    genres_to_avoid = []
    for phrase in negation_phrases:
        if phrase in user_input.lower():
            # Get the word following the negation phrase, assuming it's a genre
            genre_to_avoid = user_input.lower().split(phrase)[1].strip().split()[0]
            genres_to_avoid.append(genre_to_avoid)
    
    # Filter out movies from unwanted genres
    final_recommendations = []
    for rec in top_results:
        movie_genres = df.iloc[rec]['genres'].lower().split("|")
        if not any(genre in genres_to_avoid for genre in movie_genres):
            # Generate a list of numbered recommendations
            final_recommendations.append(f"{len(final_recommendations)+1}. {df.iloc[rec]['title']}")

            
    return "\n".join(final_recommendations)  # Return as a numbered list

examples = [
    ['I\'m in the mood for a comedy.'], 
    ['How about some action?'], 
    ['I want to watch a romance movie.']
]

iface = gr.Interface(fn=recommend_movies, 
                     inputs=gr.inputs.Textbox(lines=2, placeholder='Type something...'), 
                     outputs=gr.outputs.Textbox(),
                     examples=examples)  # Include examples
iface.launch()

