{
  "cells": [
    {
      "cell_type": "markdown",
      "metadata": {
        "id": "view-in-github",
        "colab_type": "text"
      },
      "source": [
        "<a href=\"https://colab.research.google.com/github/qunash/stable-diffusion-2-gui/blob/main/stable_diffusion_2_0.ipynb\" target=\"_parent\"><img src=\"https://colab.research.google.com/assets/colab-badge.svg\" alt=\"Open In Colab\"/></a>"
      ]
    },
    {
      "cell_type": "markdown",
      "metadata": {
        "id": "620o1BxdNbgq"
      },
      "source": [
        "# **Stable Diffusion 2.1**\n",
        "Gradio app for [Stable Diffusion 2](https://huggingface.co/stabilityai/stable-diffusion-2) by [Stability AI](https://stability.ai/) (v2-1_768-ema-pruned.ckpt).\n",
        "It uses [Hugging Face](https://huggingface.co/) Diffusers🧨 implementation.\n",
        "\n",
        "Currently supported pipelines are `text-to-image`, `image-to-image`, `inpainting`, `4x upscaling` and `depth-to-image`.\n",
        "\n",
        "<br>\n",
        "\n",
        "Colab by [anzorq](https://twitter.com/hahahahohohe). If you like it, please consider supporting me:\n",
        "\n",
        "[<a href=\"https://www.buymeacoffee.com/anzorq\" target=\"_blank\"><img src=\"https://cdn.buymeacoffee.com/buttons/v2/default-yellow.png\" height=\"32px\" width=\"108px\" alt=\"Buy Me A Coffee\"></a>](https://www.buymeacoffee.com/anzorq)\n",
        "<br>\n",
        "[![GitHub Repo stars](https://img.shields.io/github/stars/qunash/stable-diffusion-2-gui?style=social)](https://github.com/qunash/stable-diffusion-2-gui)\n",
        "\n",
        "![visitors](https://visitor-badge.glitch.me/badge?page_id=anzorq.sd-2-colab-header)"
      ]
    },
    {
      "cell_type": "markdown",
      "metadata": {
        "id": "KQI4RX20DW_8"
      },
      "source": [
        "# Install dependencies (~1.5 mins)"
      ]
    },
    {
      "cell_type": "code",
      "execution_count": 1,
      "metadata": {
        "id": "78HoqRAB-cES",
        "cellView": "form"
      },
      "outputs": [],
      "source": [
        "!pip install --upgrade git+https://github.com/huggingface/diffusers.git\n",
        "# !pip install diffusers\n",
        "!pip install --upgrade git+https://github.com/huggingface/transformers/\n",
        "# !pip install transformers\n",
        "!pip install accelerate==0.12.0\n",
        "!pip install scipy\n",
        "!pip install ftfy\n",
        "!pip install gradio -q\n",
        "\n",
        "#@markdown ### ⬅️ Run this cell\n",
        "#@markdown ---\n",
        "#@markdown ### Install **xformers**?\n",
        "#@markdown This will take an additional ~3.5 mins.<br>But images will generate 25-40% faster.\n",
        "install_xformers = False #@param {type:\"boolean\"}\n",
        "\n",
        "if install_xformers:\n",
        "  import os\n",
        "  from subprocess import getoutput\n",
        "\n",
        "  os.system(\"pip install --extra-index-url https://download.pytorch.org/whl/cu113 torch torchvision==0.13.1+cu113\")\n",
        "  os.system(\"pip install triton==2.0.0.dev20220701\")\n",
        "  gpu_info = getoutput('nvidia-smi')\n",
        "  if(\"A10G\" in gpu_info):\n",
        "      os.system(f\"pip install -q https://github.com/camenduru/stable-diffusion-webui-colab/releases/download/0.0.15/xformers-0.0.15.dev0+4c06c79.d20221205-cp38-cp38-linux_x86_64.whl\")\n",
        "  elif(\"T4\" in gpu_info):\n",
        "      os.system(f\"pip install -q https://github.com/camenduru/stable-diffusion-webui-colab/releases/download/0.0.15/xformers-0.0.15.dev0+1515f77.d20221130-cp38-cp38-linux_x86_64.whl\")\n",
        "\n",
        "\n",
        "# ### install xformers\n",
        "# from IPython.utils import capture\n",
        "# from subprocess import getoutput\n",
        "# from re import search\n",
        "\n",
        "# with capture.capture_output() as cap:\n",
        "    \n",
        "#     smi_out = getoutput('nvidia-smi')\n",
        "#     supported = search('(T4|P100|V100|A100|K80)', smi_out)\n",
        "\n",
        "#     if not supported:\n",
        "#       while True:\n",
        "#         print(\"\\x1b[1;31mThe current GPU is not supported, try starting a new session.\\x1b[0m\")\n",
        "#     else:\n",
        "#       supported = supported.group(0)\n",
        "\n",
        "# !pip install -q https://github.com/TheLastBen/fast-stable-diffusion/raw/main/precompiled/{supported}/xformers-0.0.13.dev0-py3-none-any.whl\n",
        "# !pip install -q https://github.com/ShivamShrirao/xformers-wheels/releases/download/4c06c79/xformers-0.0.15.dev0+4c06c79.d20221201-cp38-cp38-linux_x86_64.whl"
      ]
    },
    {
      "cell_type": "markdown",
      "metadata": {
        "id": "OOPHNsFYDbc0"
      },
      "source": [
        "# Run the app"
      ]
    },
    {
      "cell_type": "code",
      "execution_count": 1,
      "metadata": {
        "cellView": "form",
        "id": "gId0-asCBVwL"
      },
      "outputs": [],
      "source": [
        "#@title ⬇️🖼️\n",
        "from diffusers import StableDiffusionPipeline, StableDiffusionImg2ImgPipeline, StableDiffusionUpscalePipeline, DiffusionPipeline, StableDiffusionDepth2ImgPipeline, DPMSolverMultistepScheduler\n",
        "import gradio as gr\n",
        "import torch\n",
        "from PIL import Image\n",
        "import random\n",
        "\n",
        "state = None\n",
        "current_steps = 25\n",
        "attn_slicing_enabled = True\n",
        "mem_eff_attn_enabled = install_xformers\n",
        "\n",
        "# model_id = 'stabilityai/stable-diffusion-2'\n",
        "model_id = 'stabilityai/stable-diffusion-2-1'\n",
        "\n",
        "scheduler = DPMSolverMultistepScheduler.from_pretrained(model_id, subfolder=\"scheduler\")\n",
        "\n",
        "pipe = StableDiffusionPipeline.from_pretrained(\n",
        "      model_id,\n",
        "      revision=\"fp16\" if torch.cuda.is_available() else \"fp32\",\n",
        "      torch_dtype=torch.float16 if torch.cuda.is_available() else torch.float32,\n",
        "      scheduler=scheduler\n",
        "    ).to(\"cuda\")\n",
        "pipe.enable_attention_slicing()\n",
        "if mem_eff_attn_enabled:\n",
        "  pipe.enable_xformers_memory_efficient_attention()\n",
        "\n",
        "pipe_i2i = None\n",
        "pipe_upscale = None\n",
        "pipe_inpaint = None\n",
        "pipe_depth2img = None\n",
        "\n",
        "\n",
        "modes = {\n",
        "    'txt2img': 'Text to Image',\n",
        "    'img2img': 'Image to Image',\n",
        "    'inpaint': 'Inpainting',\n",
        "    'upscale4x': 'Upscale 4x',\n",
        "    'depth2img': 'Depth to Image'\n",
        "}\n",
        "current_mode = modes['txt2img']\n",
        "\n",
        "def error_str(error, title=\"Error\"):\n",
        "    return f\"\"\"#### {title}\n",
        "            {error}\"\"\"  if error else \"\"\n",
        "\n",
        "def update_state(new_state):\n",
        "  global state\n",
        "  state = new_state\n",
        "\n",
        "def update_state_info(old_state):\n",
        "  if state and state != old_state:\n",
        "    return gr.update(value=state)\n",
        "\n",
        "def set_mem_optimizations(pipe):\n",
        "    if attn_slicing_enabled:\n",
        "      pipe.enable_attention_slicing()\n",
        "    else:\n",
        "      pipe.disable_attention_slicing()\n",
        "    \n",
        "    if mem_eff_attn_enabled:\n",
        "      pipe.enable_xformers_memory_efficient_attention()\n",
        "    else:\n",
        "      pipe.disable_xformers_memory_efficient_attention()\n",
        "\n",
        "def get_i2i_pipe(scheduler):\n",
        "    \n",
        "    update_state(\"Loading image to image model...\")\n",
        "\n",
        "    pipe = StableDiffusionImg2ImgPipeline.from_pretrained(\n",
        "      model_id,\n",
        "      revision=\"fp16\" if torch.cuda.is_available() else \"fp32\",\n",
        "      torch_dtype=torch.float16 if torch.cuda.is_available() else torch.float32,\n",
        "      scheduler=scheduler\n",
        "    )\n",
        "    set_mem_optimizations(pipe)\n",
        "    pipe.to(\"cuda\")\n",
        "    return pipe\n",
        "\n",
        "def get_inpaint_pipe():\n",
        "  \n",
        "  update_state(\"Loading inpainting model...\")\n",
        "\n",
        "  pipe = DiffusionPipeline.from_pretrained(\n",
        "      \"stabilityai/stable-diffusion-2-inpainting\",\n",
        "      revision=\"fp16\" if torch.cuda.is_available() else \"fp32\",\n",
        "      torch_dtype=torch.float16 if torch.cuda.is_available() else torch.float32,\n",
        "      # scheduler=scheduler # TODO currently setting scheduler here messes up the end result. A bug in Diffusers🧨\n",
        "    ).to(\"cuda\")\n",
        "  pipe.scheduler = DPMSolverMultistepScheduler.from_config(pipe.scheduler.config)\n",
        "  pipe.enable_attention_slicing()\n",
        "  pipe.enable_xformers_memory_efficient_attention()\n",
        "  return pipe\n",
        "\n",
        "def get_upscale_pipe(scheduler):\n",
        "    \n",
        "    update_state(\"Loading upscale model...\")\n",
        "\n",
        "    pipe = StableDiffusionUpscalePipeline.from_pretrained(\n",
        "      \"stabilityai/stable-diffusion-x4-upscaler\",\n",
        "      revision=\"fp16\" if torch.cuda.is_available() else \"fp32\",\n",
        "      torch_dtype=torch.float16 if torch.cuda.is_available() else torch.float32,\n",
        "      # scheduler=scheduler\n",
        "    )\n",
        "    # pipe.scheduler = DPMSolverMultistepScheduler.from_config(pipe.scheduler.config)\n",
        "    set_mem_optimizations(pipe)\n",
        "    pipe.to(\"cuda\")\n",
        "    return pipe\n",
        "    \n",
        "def get_depth2img_pipe():\n",
        "    \n",
        "    update_state(\"Loading depth to image model...\")\n",
        "\n",
        "    pipe = StableDiffusionDepth2ImgPipeline.from_pretrained(\n",
        "     \"stabilityai/stable-diffusion-2-depth\",\n",
        "      revision=\"fp16\" if torch.cuda.is_available() else \"fp32\",\n",
        "      torch_dtype=torch.float16 if torch.cuda.is_available() else torch.float32,\n",
        "      # scheduler=scheduler\n",
        "    )\n",
        "    pipe.scheduler = DPMSolverMultistepScheduler.from_config(pipe.scheduler.config)\n",
        "    set_mem_optimizations(pipe)\n",
        "    pipe.to(\"cuda\")\n",
        "    return pipe\n",
        "\n",
        "def switch_attention_slicing(attn_slicing):\n",
        "    global attn_slicing_enabled\n",
        "    attn_slicing_enabled = attn_slicing\n",
        "\n",
        "def switch_mem_eff_attn(mem_eff_attn):\n",
        "    global mem_eff_attn_enabled\n",
        "    mem_eff_attn_enabled = mem_eff_attn\n",
        "\n",
        "def pipe_callback(step: int, timestep: int, latents: torch.FloatTensor):\n",
        "    update_state(f\"{step}/{current_steps} steps\")#\\nTime left, sec: {timestep/100:.0f}\")\n",
        "\n",
        "def inference(inf_mode, prompt, n_images, guidance, steps, width=768, height=768, seed=0, img=None, strength=0.5, neg_prompt=\"\"):\n",
        "\n",
        "  update_state(\" \")\n",
        "\n",
        "  global current_mode\n",
        "  if inf_mode != current_mode:\n",
        "    pipe.to(\"cuda\" if inf_mode == modes['txt2img'] else \"cpu\")\n",
        "\n",
        "    if pipe_i2i is not None:\n",
        "      pipe_i2i.to(\"cuda\" if inf_mode == modes['img2img'] else \"cpu\")\n",
        "\n",
        "    if pipe_inpaint is not None:\n",
        "      pipe_inpaint.to(\"cuda\" if inf_mode == modes['inpaint'] else \"cpu\")\n",
        "\n",
        "    if pipe_upscale is not None:\n",
        "      pipe_upscale.to(\"cuda\" if inf_mode == modes['upscale4x'] else \"cpu\")\n",
        "    \n",
        "    if pipe_depth2img is not None:\n",
        "      pipe_depth2img.to(\"cuda\" if inf_mode == modes['depth2img'] else \"cpu\")\n",
        "\n",
        "    current_mode = inf_mode\n",
        "    \n",
        "  if seed == 0:\n",
        "    seed = random.randint(0, 2147483647)\n",
        "\n",
        "  generator = torch.Generator('cuda').manual_seed(seed)\n",
        "  prompt = prompt\n",
        "\n",
        "  try:\n",
        "    \n",
        "    if inf_mode == modes['txt2img']:\n",
        "      return txt_to_img(prompt, n_images, neg_prompt, guidance, steps, width, height, generator, seed), gr.update(visible=False, value=None)\n",
        "    \n",
        "    elif inf_mode == modes['img2img']:\n",
        "      if img is None:\n",
        "        return None, gr.update(visible=True, value=error_str(\"Image is required for Image to Image mode\"))\n",
        "\n",
        "      return img_to_img(prompt, n_images, neg_prompt, img, strength, guidance, steps, width, height, generator, seed), gr.update(visible=False, value=None)\n",
        "    \n",
        "    elif inf_mode == modes['inpaint']:\n",
        "      if img is None:\n",
        "        return None, gr.update(visible=True, value=error_str(\"Image is required for Inpainting mode\"))\n",
        "\n",
        "      return inpaint(prompt, n_images, neg_prompt, img, guidance, steps, width, height, generator, seed), gr.update(visible=False, value=None)\n",
        "\n",
        "    elif inf_mode == modes['upscale4x']:\n",
        "      if img is None:\n",
        "        return None, gr.update(visible=True, value=error_str(\"Image is required for Upscale mode\"))\n",
        "\n",
        "      return upscale(prompt, n_images, neg_prompt, img, guidance, steps, generator), gr.update(visible=False, value=None)\n",
        "\n",
        "    elif inf_mode == modes['depth2img']:\n",
        "      if img is None:\n",
        "        return None, gr.update(visible=True, value=error_str(\"Image is required for Depth to Image mode\"))\n",
        "\n",
        "      return depth2img(prompt, n_images, neg_prompt, img, guidance, steps, generator, seed), gr.update(visible=False, value=None)\n",
        "\n",
        "  except Exception as e:\n",
        "    return None, gr.update(visible=True, value=error_str(e))\n",
        "\n",
        "def txt_to_img(prompt, n_images, neg_prompt, guidance, steps, width, height, generator, seed):\n",
        "\n",
        "    result = pipe(\n",
        "      prompt,\n",
        "      num_images_per_prompt = n_images,\n",
        "      negative_prompt = neg_prompt,\n",
        "      num_inference_steps = int(steps),\n",
        "      guidance_scale = guidance,\n",
        "      width = width,\n",
        "      height = height,\n",
        "      generator = generator,\n",
        "      callback=pipe_callback).images\n",
        "\n",
        "    update_state(f\"Done. Seed: {seed}\")\n",
        "\n",
        "    return result\n",
        "\n",
        "def img_to_img(prompt, n_images, neg_prompt, img, strength, guidance, steps, width, height, generator, seed):\n",
        "\n",
        "    global pipe_i2i\n",
        "    if pipe_i2i is None:\n",
        "      pipe_i2i = get_i2i_pipe(scheduler)\n",
        "\n",
        "    img = img['image']\n",
        "    ratio = min(height / img.height, width / img.width)\n",
        "    img = img.resize((int(img.width * ratio), int(img.height * ratio)), Image.LANCZOS)\n",
        "    result = pipe_i2i(\n",
        "      prompt,\n",
        "      num_images_per_prompt = n_images,\n",
        "      negative_prompt = neg_prompt,\n",
        "      image = img,\n",
        "      num_inference_steps = int(steps),\n",
        "      strength = strength,\n",
        "      guidance_scale = guidance,\n",
        "      # width = width,\n",
        "      # height = height,\n",
        "      generator = generator,\n",
        "      callback=pipe_callback).images\n",
        "\n",
        "    update_state(f\"Done. Seed: {seed}\")\n",
        "        \n",
        "    return result\n",
        "\n",
        "# TODO Currently supports only 512x512 images\n",
        "def inpaint(prompt, n_images, neg_prompt, img, guidance, steps, width, height, generator, seed):\n",
        "\n",
        "    global pipe_inpaint\n",
        "    if pipe_inpaint is None:\n",
        "      pipe_inpaint = get_inpaint_pipe()\n",
        "\n",
        "    inp_img = img['image']\n",
        "    mask = img['mask']\n",
        "    inp_img = square_padding(inp_img)\n",
        "    mask = square_padding(mask)\n",
        "\n",
        "    # # ratio = min(height / inp_img.height, width / inp_img.width)\n",
        "    # ratio = min(512 / inp_img.height, 512 / inp_img.width)\n",
        "    # inp_img = inp_img.resize((int(inp_img.width * ratio), int(inp_img.height * ratio)), Image.LANCZOS)\n",
        "    # mask = mask.resize((int(mask.width * ratio), int(mask.height * ratio)), Image.LANCZOS)\n",
        "\n",
        "    inp_img = inp_img.resize((512, 512))\n",
        "    mask = mask.resize((512, 512))\n",
        "\n",
        "    result = pipe_inpaint(\n",
        "      prompt,\n",
        "      image = inp_img,\n",
        "      mask_image = mask,\n",
        "      num_images_per_prompt = n_images,\n",
        "      negative_prompt = neg_prompt,\n",
        "      num_inference_steps = int(steps),\n",
        "      guidance_scale = guidance,\n",
        "      # width = width,\n",
        "      # height = height,\n",
        "      generator = generator,\n",
        "      callback=pipe_callback).images\n",
        "        \n",
        "    update_state(f\"Done. Seed: {seed}\")\n",
        "\n",
        "    return result\n",
        "\n",
        "def depth2img(prompt, n_images, neg_prompt, img, guidance, steps, generator, seed):\n",
        "\n",
        "    global pipe_depth2img\n",
        "    if pipe_depth2img is None:\n",
        "      pipe_depth2img = get_depth2img_pipe()\n",
        "\n",
        "    img = img['image']\n",
        "    result = pipe_depth2img(\n",
        "      prompt,\n",
        "      num_images_per_prompt = n_images,\n",
        "      negative_prompt = neg_prompt,\n",
        "      image = img,\n",
        "      num_inference_steps = int(steps),\n",
        "      guidance_scale = guidance,\n",
        "      # width = width,\n",
        "      # height = height,\n",
        "      generator = generator,\n",
        "      callback=pipe_callback).images\n",
        "\n",
        "    update_state(f\"Done. Seed: {seed}\")\n",
        "        \n",
        "    return result\n",
        "\n",
        "def square_padding(img):\n",
        "    width, height = img.size\n",
        "    if width == height:\n",
        "        return img\n",
        "    new_size = max(width, height)\n",
        "    new_img = Image.new('RGB', (new_size, new_size), (0, 0, 0, 255))\n",
        "    new_img.paste(img, ((new_size - width) // 2, (new_size - height) // 2))\n",
        "    return new_img\n",
        "\n",
        "def upscale(prompt, n_images, neg_prompt, img, guidance, steps, generator):\n",
        "\n",
        "    global pipe_upscale\n",
        "    if pipe_upscale is None:\n",
        "      pipe_upscale = get_upscale_pipe(scheduler)\n",
        "\n",
        "    img = img['image']\n",
        "    return upscale_tiling(prompt, neg_prompt, img, guidance, steps, generator)\n",
        "\n",
        "    # result = pipe_upscale(\n",
        "    #     prompt,\n",
        "    #     image = img,\n",
        "    #     num_inference_steps = int(steps),\n",
        "    #     guidance_scale = guidance,\n",
        "    #     negative_prompt = neg_prompt,\n",
        "    #     num_images_per_prompt = n_images,\n",
        "    #     generator = generator).images[0]\n",
        "\n",
        "    # return result\n",
        "\n",
        "def upscale_tiling(prompt, neg_prompt, img, guidance, steps, generator):\n",
        "\n",
        "    width, height = img.size\n",
        "\n",
        "    # calculate the padding needed to make the image dimensions a multiple of 128\n",
        "    padding_x = 128 - (width % 128) if width % 128 != 0 else 0\n",
        "    padding_y = 128 - (height % 128) if height % 128 != 0 else 0\n",
        "\n",
        "    # create a white image of the right size to be used as padding\n",
        "    padding_img = Image.new('RGB', (padding_x, padding_y), color=(255, 255, 255, 0))\n",
        "\n",
        "    # paste the padding image onto the original image to add the padding\n",
        "    img.paste(padding_img, (width, height))\n",
        "\n",
        "    # update the image dimensions to include the padding\n",
        "    width += padding_x\n",
        "    height += padding_y\n",
        "\n",
        "    if width > 128 or height > 128:\n",
        "\n",
        "        num_tiles_x = int(width / 128)\n",
        "        num_tiles_y = int(height / 128)\n",
        "\n",
        "        upscaled_img = Image.new('RGB', (img.size[0] * 4, img.size[1] * 4))\n",
        "        for x in range(num_tiles_x):\n",
        "            for y in range(num_tiles_y):\n",
        "                update_state(f\"Upscaling tile {x * num_tiles_y + y + 1}/{num_tiles_x * num_tiles_y}\")\n",
        "                tile = img.crop((x * 128, y * 128, (x + 1) * 128, (y + 1) * 128))\n",
        "\n",
        "                upscaled_tile = pipe_upscale(\n",
        "                    prompt=\"\",\n",
        "                    image=tile,\n",
        "                    num_inference_steps=steps,\n",
        "                    guidance_scale=guidance,\n",
        "                    # negative_prompt = neg_prompt,\n",
        "                    generator=generator,\n",
        "                ).images[0]\n",
        "\n",
        "                upscaled_img.paste(upscaled_tile, (x * upscaled_tile.size[0], y * upscaled_tile.size[1]))\n",
        "\n",
        "        return [upscaled_img]\n",
        "    else:\n",
        "        return pipe_upscale(\n",
        "            prompt=prompt,\n",
        "            image=img,\n",
        "            num_inference_steps=steps,\n",
        "            guidance_scale=guidance,\n",
        "            negative_prompt = neg_prompt,\n",
        "            generator=generator,\n",
        "        ).images\n",
        "\n",
        "\n",
        "\n",
        "def on_mode_change(mode):\n",
        "  return gr.update(visible = mode in (modes['img2img'], modes['inpaint'], modes['upscale4x'], modes['depth2img'])), \\\n",
        "         gr.update(visible = mode == modes['inpaint']), \\\n",
        "         gr.update(visible = mode == modes['upscale4x']), \\\n",
        "         gr.update(visible = mode == modes['img2img'])\n",
        "\n",
        "def on_steps_change(steps):\n",
        "  global current_steps\n",
        "  current_steps = steps\n",
        "\n",
        "css = \"\"\".main-div div{display:inline-flex;align-items:center;gap:.8rem;font-size:1.75rem}.main-div div h1{font-weight:900;margin-bottom:7px}.main-div p{margin-bottom:10px;font-size:94%}a{text-decoration:underline}.tabs{margin-top:0;margin-bottom:0}#gallery{min-height:20rem}\n",
        "\"\"\"\n",
        "with gr.Blocks(css=css) as demo:\n",
        "    gr.HTML(\n",
        "        f\"\"\"\n",
        "          <div class=\"main-div\">\n",
        "            <div>\n",
        "              <h1>Stable Diffusion 2.1</h1>\n",
        "            </div><br>\n",
        "            <p> Model used: <a href=\"https://huggingface.co/stabilityai/stable-diffusion-2-1/blob/main/v2-1_768-ema-pruned.ckpt\" target=\"_blank\">v2-1_768-ema-pruned.ckpt</a></p>\n",
        "            Running on <b>{\"GPU 🔥\" if torch.cuda.is_available() else \"CPU 🥶\"}</b>\n",
        "          </div>\n",
        "        \"\"\"\n",
        "    )\n",
        "    with gr.Row():\n",
        "        \n",
        "        with gr.Column(scale=70):\n",
        "          with gr.Group():\n",
        "              with gr.Row():\n",
        "                prompt = gr.Textbox(label=\"Prompt\", show_label=False, max_lines=2,placeholder=f\"Enter prompt\").style(container=False)\n",
        "                generate = gr.Button(value=\"Generate\").style(rounded=(False, True, True, False))\n",
        "\n",
        "              gallery = gr.Gallery(label=\"Generated images\", show_label=False).style(grid=[2], height=\"auto\")\n",
        "          state_info = gr.Textbox(label=\"State\", show_label=False, max_lines=2).style(container=False)\n",
        "          error_output = gr.Markdown(visible=False)\n",
        "\n",
        "        with gr.Column(scale=30):\n",
        "          inf_mode = gr.Radio(label=\"Inference Mode\", choices=list(modes.values()), value=modes['txt2img'])\n",
        "          \n",
        "          with gr.Group(visible=False) as i2i_options:\n",
        "            image = gr.Image(label=\"Image\", height=128, type=\"pil\", tool='sketch')\n",
        "            inpaint_info = gr.Markdown(\"Inpainting resizes and pads images to 512x512\", visible=False)\n",
        "            upscale_info = gr.Markdown(\"\"\"Best for small images (128x128 or smaller).<br>\n",
        "                                        Bigger images will be sliced into 128x128 tiles which will be upscaled individually.<br>\n",
        "                                        This is done to avoid running out of GPU memory.\"\"\", visible=False)\n",
        "            strength = gr.Slider(label=\"Transformation strength\", minimum=0, maximum=1, step=0.01, value=0.5)\n",
        "\n",
        "          with gr.Group():\n",
        "            neg_prompt = gr.Textbox(label=\"Negative prompt\", placeholder=\"What to exclude from the image\")\n",
        "\n",
        "            n_images = gr.Slider(label=\"Number of images\", value=1, minimum=1, maximum=4, step=1)\n",
        "            with gr.Row():\n",
        "              guidance = gr.Slider(label=\"Guidance scale\", value=7.5, maximum=15)\n",
        "              steps = gr.Slider(label=\"Steps\", value=current_steps, minimum=2, maximum=100, step=1)\n",
        "\n",
        "            with gr.Row():\n",
        "              width = gr.Slider(label=\"Width\", value=768, minimum=64, maximum=1024, step=8)\n",
        "              height = gr.Slider(label=\"Height\", value=768, minimum=64, maximum=1024, step=8)\n",
        "\n",
        "            seed = gr.Slider(0, 2147483647, label='Seed (0 = random)', value=0, step=1)\n",
        "            with gr.Accordion(\"Memory optimization\"):\n",
        "              attn_slicing = gr.Checkbox(label=\"Attention slicing (a bit slower, but uses less memory)\", value=attn_slicing_enabled)\n",
        "              # mem_eff_attn = gr.Checkbox(label=\"Memory efficient attention (xformers)\", value=mem_eff_attn_enabled)\n",
        "\n",
        "    inf_mode.change(on_mode_change, inputs=[inf_mode], outputs=[i2i_options, inpaint_info, upscale_info, strength], queue=False)\n",
        "    steps.change(on_steps_change, inputs=[steps], outputs=[], queue=False)\n",
        "    attn_slicing.change(lambda x: switch_attention_slicing(x), inputs=[attn_slicing], queue=False)\n",
        "    # mem_eff_attn.change(lambda x: switch_mem_eff_attn(x), inputs=[mem_eff_attn], queue=False)\n",
        "\n",
        "    inputs = [inf_mode, prompt, n_images, guidance, steps, width, height, seed, image, strength, neg_prompt]\n",
        "    outputs = [gallery, error_output]\n",
        "    prompt.submit(inference, inputs=inputs, outputs=outputs)\n",
        "    generate.click(inference, inputs=inputs, outputs=outputs)\n",
        "\n",
        "    demo.load(update_state_info, inputs=state_info, outputs=state_info, every=0.5, show_progress=False)\n",
        "\n",
        "    gr.HTML(\"\"\"\n",
        "    <div style=\"border-top: 1px solid #303030;\">\n",
        "      <br>\n",
        "      <p>Space by: <a href=\"https://twitter.com/hahahahohohe\"><img src=\"https://img.shields.io/twitter/follow/hahahahohohe?label=%40anzorq&style=social\" alt=\"Twitter Follow\"></a></p><br>\n",
        "      <p>Enjoying this app? Please consider <a href=\"https://www.buymeacoffee.com/anzorq\">supporting me</a></p>\n",
        "      <a href=\"https://www.buymeacoffee.com/anzorq\" target=\"_blank\"><img src=\"https://cdn.buymeacoffee.com/buttons/v2/default-yellow.png\" alt=\"Buy Me A Coffee\" style=\"height: 45px !important;width: 162px !important;\" ></a><br><br>\n",
        "      <a href=\"https://github.com/qunash/stable-diffusion-2-gui\" target=\"_blank\"><img alt=\"GitHub Repo stars\" src=\"https://img.shields.io/github/stars/qunash/stable-diffusion-2-gui?style=social\"></a>\n",
        "      <p><img src=\"https://visitor-badge.glitch.me/badge?page_id=anzorq.sd-2-colab\" alt=\"visitors\"></p>\n",
        "    </div>\n",
        "    \"\"\")\n",
        "\n",
        "demo.queue()\n",
        "demo.launch(debug=True, share=True, height=768)\n"
      ]
    }
  ],
  "metadata": {
    "accelerator": "GPU",
    "colab": {
      "private_outputs": True,
      "provenance": [],
      "toc_visible": True,
      "include_colab_link": False
    },
    "gpuClass": "standard",
    "kernelspec": {
      "display_name": "Python 3",
      "name": "python3"
    },
    "language_info": {
      "name": "python"
    }
  },
  "nbformat": 4,
  "nbformat_minor": 0
}