import gradio as gr

from transformers import VisionEncoderDecoderModel, ViTImageProcessor, AutoTokenizer
from transformers import CLIPProcessor, CLIPModel
from transformers import pipeline

import torch
from PIL import Image

import openai
import ai_functions

import re
import os

HF_TOKEN = os.environ.get("HF_TOKEN")

device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

model = VisionEncoderDecoderModel.from_pretrained("nlpconnect/vit-gpt2-image-captioning")
feature_extractor = ViTImageProcessor.from_pretrained("nlpconnect/vit-gpt2-image-captioning")
tokenizer = AutoTokenizer.from_pretrained("nlpconnect/vit-gpt2-image-captioning")

clip = CLIPModel.from_pretrained("openai/clip-vit-large-patch14")
processor = CLIPProcessor.from_pretrained("openai/clip-vit-large-patch14")

clip = clip.to(device)

llm = None

model.to(device)

max_length = 16
num_beams = 4
gen_kwargs = {"max_length": max_length, "num_beams": num_beams}

@staticmethod
def extract_list_items(text):
	pattern = r"\d+[.\)-](?=\D)"
	return re.split(pattern, text)[1:]

def process_image_and_text(image, text, model_choice, api_key=None):
	global llm
	if model_choice != "llama 7b":
		openai.api_key = api_key
	
	pixel_values = feature_extractor(images=image, return_tensors="pt").pixel_values
	pixel_values = pixel_values.to(device)

	output_ids = model.generate(pixel_values, **gen_kwargs)

	preds = tokenizer.batch_decode(output_ids, skip_special_tokens=True)
	preds = [pred.strip() for pred in preds][0]

	if model_choice == "llama 7b":
		result = ai_functions.ai_function(text, preds, llm)
	else:
		result = ai_functions.ai_function(text, preds, model_choice)

	prompt = extract_list_items(result)

	inputs = processor(text=prompt, images=image, return_tensors="pt", padding=True)
	inputs = {name: tensor.to(device) for name, tensor in inputs.items()}

	outputs = clip(**inputs)
	logits_per_image = outputs.logits_per_image
	probs = logits_per_image.softmax(dim=1)

	return prompt[torch.argmax(torch.FloatTensor(probs.cpu()))]

model_choices = gr.Dropdown(choices=["gpt-4", "gpt-3.5-turbo"], label="Model")
api_key_input = gr.Textbox(label="OpenAI API Key")

with gr.Blocks() as demo:
    gr.Markdown("# LLM Can See\n\nThis is an integration project using the Large Language Model(LLM), vit-gpt2, and CLIP, with the aim of enabling LLM to interpret images and answer related questions.")
    
    
    with gr.Row():
        gr.Interface(
            fn=process_image_and_text, 
            inputs=["image", "text", model_choices, api_key_input],
            outputs=["text"],
            allow_flagging="never"
        )
    gr.Markdown("## How to Use\n\n1. **Upload an image**: Choose an image you want the model to interpret.\n2. **Enter a question**: Input your question for the model in the text box.\n3. **Select a model**: Select the model you wish to use for the answer. The current supported models are \"gpt-4\" and \"gpt-3.5-turbo\". Due to memory limitations, the \"llama2\" model is not currently supported.\n\n## Notes\n\nThe current prompt has been preset to \"assist the visually impaired\", so the questions answered by the model will be particularly biased towards this setting. Please note that this may affect the style and content of the model's answers.\n\n")

demo.launch()