import cv2
import gradio as gr
import easyocr
import numpy as np
import json
from collections import OrderedDict
from pprint import pformat

try:
    import tomllib
except ModuleNotFoundError:
    import tomli as tomllib


def read_file(path: str) -> str:
    with open(path, "rb") as f:
        return f.read().decode("utf-8")


def draw_label(
    image,
    center,
    text,
    font=cv2.FONT_HERSHEY_SIMPLEX,
    font_scale=1,
    text_color=(255, 0, 0),
    text_thickness=2,
    circle_color=(0, 0, 0),
    circle_thickness=-1,
):
    (text_width, text_height), _ = cv2.getTextSize(
        str(text), font, font_scale, text_thickness
    )
    text_x = center[0] - text_width // 2
    text_y = center[1] + text_height // 2
    radius = (text_width + 8) // 2
    cv2.circle(image, center, radius, circle_color, circle_thickness)
    cv2.putText(
        image, str(text), (text_x, text_y), font, font_scale, text_color, text_thickness
    )
    cv2.circle(image, center, radius + 1, text_color, 1)


class Data:
    @staticmethod
    def block_factory(block_metadata: list[dict]):
        _block_list = [
            (bm["name"], getattr(gr, bm["type"])(render=False, **bm["param"]))
            for bm in block_metadata
        ]
        return OrderedDict(_block_list)

    def __init__(
        self, toml_file: str, batch_size: int = 1, workers: int = 0, canvas_size=2560
    ):
        with open(toml_file, "rb") as f:
            _data = tomllib.load(f, parse_float=float)
        self.assets = {k: read_file(v) for k, v in _data["assets"]}
        self.inputs = Data.block_factory(_data["blocklist"]["inputs"])
        self.default = [getattr(b, "value", None) for b in self.inputs.values()]
        self.outputs = Data.block_factory(_data["blocklist"]["outputs"])
        self.examples = _data["examples"]
        self.batch_size = batch_size
        self.workers = workers
        self.canvas_size = canvas_size

    def render(self, *args):
        for bn in args:
            block = self.inputs.get(bn, self.outputs.get(bn))
            if block:
                block.render()

    @property
    def inputs_list(self):
        return list(self.inputs.values())

    @property
    def outputs_list(self):
        return list(self.outputs.values())

    def process_image(self, lang: list[str], *args):
        reader = easyocr.Reader(lang, False)
        kwargs = dict(zip(list(self.inputs.keys())[1:], args))
        kwargs.update(
            {
                "batch_size": self.batch_size,
                "workers": self.workers,
                "canvas_size": self.canvas_size,
            }
        )
        raw = reader.readtext(**kwargs)
        result = raw.copy()
        raw = pformat(raw)
        if not kwargs["detail"]:
            return kwargs["image"], [[idx, i, ""] for idx, i in enumerate(result)], raw

        if kwargs["paragraph"]:
            result = [[i[0], i[1], 0] for i in result]
        if kwargs["output_format"] == "json":
            result = [json.loads(i, parse_float=float, parse_int=int) for i in result]
        if kwargs["output_format"] == "dict" or kwargs["output_format"] == "json":
            result = [[i["boxes"], i["text"], i["confident"]] for i in result]

        cnt = [np.array(i[0]).astype(int) for i in result]
        cv2.drawContours(kwargs["image"], cnt, -1, (0, 255, 0), 3, cv2.LINE_AA)
        for count, c in enumerate(cnt):
            draw_label(kwargs["image"], c[0], count)
        data = [[idx, i[1], f"{int(i[2]*100)}%"] for idx, i in enumerate(result)]

        return kwargs["image"], data, raw
