import gradio as gr
import os, json
import pandas as pd
from constants import ALL_COLUMNS, DATA_TYPES, DEFAULT_CHECK, MODEL_SIZES, MODEL_TYPES, SOURCE_TYPES, DATASETS


def read_jsons(directory_name, col_names):
    json_files = [pos_json for pos_json in os.listdir(directory_name)]
    table = []
    for js in json_files:
        with open(os.path.join(directory_name, js)) as json_file:
            json_text = json.load(json_file)
            table.append(json_text)
    df = pd.DataFrame(table)
    df = df[df.columns.intersection(col_names)] # filter based on col_names
    return df



def update_cols(leaderboard: pd.DataFrame, datasets: list,):
    cols = ["Model","Type","Source-type","Size"]
    cols.extend(datasets)
    return gr.Dataframe(
        value = read_jsons(directory_name="model_data", col_names=cols),
        headers = cols,
        datatype=[DATA_TYPES[x] for x in ALL_COLUMNS],
        interactive=False,
    )



def update_rows(leaderboard: pd.DataFrame, datasets: list, types : list, source_types : list, sizes : list):

    cols = ["Model","Type","Source-type","Size"]
    cols.extend(datasets)
   
    unfiltered = read_jsons(directory_name="model_data", col_names=cols)
    filtered = filter(unfiltered, types, source_types, sizes)

    df = gr.Dataframe(
        value = filtered,
        headers = cols,
        datatype=[DATA_TYPES[x] for x in ALL_COLUMNS],
        interactive=False,
    )

    return df

def filter(df:pd.DataFrame , types : list, source_types : list, sizes : list):
    df = df[df["Size"].isin(sizes)]
    df = df[df["Source-type"].isin(source_types)]
    df = df[df["Type"].isin(types)]
    return df

with gr.Blocks() as demo:

    gr.Markdown("# Welcome to the German LLM leaderboard!")

    with gr.Row(): 
        with gr.Column():
            with gr.Row():
                parameter_size = gr.CheckboxGroup(
                choices=MODEL_SIZES,
                value=MODEL_SIZES,
                label="Model sizes",
                elem_id="size-select",
                interactive=True,
                )
                
            with gr.Row():
                model_type = gr.CheckboxGroup(
                choices=MODEL_TYPES,
                value=MODEL_TYPES,
                label="Model types",
                elem_id="type-select",
                interactive=True,
                )

        with gr.Column():
            with gr.Row():
                source_type = gr.CheckboxGroup(
                choices=SOURCE_TYPES,
                value=SOURCE_TYPES,
                label="Source types",
                elem_id="source-select",
                interactive=True,
                )

            with gr.Row():
                shown_columns = gr.CheckboxGroup(
                    choices=DATASETS,
                    value=DATASETS,
                    label="Select datasets to show",
                    elem_id="column-select",
                    interactive=True,
                    )
        

    leaderboard_table = gr.Dataframe(
        headers = ALL_COLUMNS,
        value = read_jsons(directory_name="model_data", col_names=ALL_COLUMNS),
        datatype=[DATA_TYPES[x] for x in ALL_COLUMNS],
        interactive=False,
    )

    shown_columns.change(
        fn=update_cols,
        inputs=[
            leaderboard_table, 
            shown_columns,
            ],
        outputs=leaderboard_table,
    )


    parameter_size.change(
        fn=update_rows,
        inputs=[
            leaderboard_table, 
            shown_columns,
            model_type,
            source_type,
            parameter_size,
        ],
    outputs=leaderboard_table,
    )

    source_type.change(
        fn=update_rows,
        inputs=[
            leaderboard_table, 
            shown_columns,        
            model_type,
            source_type,
            parameter_size,
        ],
    outputs=leaderboard_table,
    )

    model_type.change(
        fn=update_rows,
        inputs=[
            leaderboard_table, 
            shown_columns,
            model_type,
            source_type,
            parameter_size,
            ],
        outputs=leaderboard_table,
    )

if __name__ == "__main__":
    demo.launch()