import gradio as gr
import os 
import torch
from model import create_effnetb2_model
from timeit import default_timer as timer
from typing import Tuple, Dict

class_names = ['pizza', 'steak', 'sushi']
effnetb2, effnetb2_transforms = create_effnetb2_model(3, 42)
# Load save weights:
effnetb2.load_state_dict(
  torch.load(f='09_pretrained_effnetb2_feature_extractor_pizza_steak_sushi_20_precent.pth',
  map_location=torch.device('cpu')
  )
)

def predict(img):
  # Start a timer
  start_time = timer()
  # Transform the input image for use wit EffNetB2
  img = effnetb2_transforms(img).unsqueeze(0)
  # Put model into eval mode, make prediction
  effnetb2.eval()
  with torch.inference_mode():
    pred_probs = torch.softmax(effnetb2(img), dim=1)
  # Create a prediction labal and prediction probability dictionary
  pred_labels_and_probs = {class_names[i]:float(pred_probs[0][i]) for i in range(len(class_names))}
  # Calculated pred time
  end_time = timer()
  pred_time = round(end_time - start_time, 4)
  # Return pred dict and pred time
  return pred_labels_and_probs, pred_time


title = 'FoodVision Mini 🍕🥩🍣'
description = 'An [EfficientNetB2 feature extractor](https://pytorch.org/vision/main/models/generated/torchvision.models.efficientnet_b2.html)'
article = 'Created with Pytorch model deployment'
example_list = [["./examples/" + file] for file in os.listdir("./examples")]

demo = gr.Interface(fn=predict,
             inputs=gr.Image(type='pil'),
             outputs=[gr.Label(num_top_classes=3, label='Predictions'),
                     gr.Number(label='Prediction time (s)')],
             examples=example_list,
             title=title,
             description=description,
             article=article
             )

demo.launch(debug=False,
            share=False)
