import torch
import numpy as np
from torchvision import datasets, transforms, models
import torch.nn as nn
import torch.nn.functional as F
import gradio as gr
import PIL.Image as Image
import skimage.transform
import cv2



def load_model():
    model = models.efficientnet_b4()
    model.classifier[1] = nn.Linear(1792, 13)
    model.load_state_dict(torch.load('model.pth', map_location='cpu'))
    model.eval()
    return model


def load_labels():
    labels = open('classes.txt').read().splitlines()
    return labels

model = load_model()
labels = load_labels()

def preprocess(img):
    # img = Image.fromarray(img.astype('uint8'), 'RGB')
    r_image = transforms.Compose([transforms.Resize((380,380)),
              transforms.ToTensor(), 
              transforms.Normalize(mean = [0.485, 0.456, 0.406], std = [0.229, 0.224, 0.225])])(img)
    return r_image


class Hook():
    features=None
    def __init__(self, m): 
        self.hook = m.register_forward_hook(self.hook_fn)
    def hook_fn(self, module, input, output): 
        self.features = ((output.cpu()).data).numpy()
    def remove(self): 
        self.hook.remove()


def cam(conv_features, weights, class_idx):
    counts, c, h, w = conv_features.shape
    output_cam = []
    cam = weights[class_idx].dot(conv_features.reshape((c, h*w)))
    cam = cam.reshape(h, w)
    cam = cam - np.min(cam)
    cam_img = cam /np.max(cam)
    cam_img = np.uint8(255*cam_img)
    output_cam.append(cam_img)
    return output_cam
    

# gradio app for cam
def cam_app(img):
    img2 = img.resize((380, 380))
    img = preprocess(img)
    img = img.unsqueeze(0)
    last_layer  = model.features._modules.get("8")
    hooked_features = Hook(last_layer)
    pred = model(img)
    pred_prob = F.softmax(pred, dim = 1)
    pred_prob = pred_prob.detach().cpu().numpy()
    chosen_class = pred_prob.argmax()
    weights_fc = list(model.classifier.parameters())[-2]
    weights_fc = weights_fc.detach().cpu().numpy()
    cam_mask = cam(conv_features=hooked_features.features, weights=weights_fc, class_idx=chosen_class)
    # return the blended image
    img = np.array(img2)
    mask_arr = np.array(cam_mask[0])
    mask_arr = skimage.transform.resize(mask_arr, (380, 380))
    # match the mask to the image
    mask_arr = np.uint8(255*mask_arr)
    mask_arr = cv2.applyColorMap(mask_arr, cv2.COLORMAP_JET)
    mask_arr = cv2.cvtColor(mask_arr, cv2.COLOR_BGR2RGB)
    mask_arr = (mask_arr.astype(float))/255
    img = (img.astype(float))/255 
    blended_img = (cv2.addWeighted(img, 0.5, mask_arr, 0.5, 0))*255
    blended_img = blended_img.astype(np.uint8)
    blended_img = Image.fromarray(blended_img)

    # top 3 predictions as a percentage bar
    top3 = pred_prob.argsort()[0][-3:]
    top3 = top3[::-1]
    top3_conf = pred_prob[0][top3]
    top3_conf = top3_conf*100
    top3_conf = top3_conf.round(2)
    top3_labels = [labels[i] for i in top3]
    top3_labels = [str(i) + " : " + str(j) + "%" for i,j in zip(top3_labels, top3_conf)]
    top3_labels = " , ".join(top3_labels)
    return blended_img, top3_labels




# App
description = "Classify Kenyan food into 13 categories"
article = "<p style='text-align: center'><a href='https://github.com/ariharasudhanm/Image_classification_Kaggle_Competition'>Github</a> | <a href='https://www.linkedin.com/in/ariharasudhan/'>LinkedIn</a></p>"
examples = [ "./Test_Images/unknown2.jpg", "./Test_Images/unknown3.jpg", "./Test_Images/unknown5.jpg"]
gr.Interface(cam_app,
                        inputs=gr.inputs.Image( type = "pil", label="Input Image"),
                        outputs=[gr.outputs.components.Image(type = "pil", label="XAI-Class Activation Map").style(height = 300, width = 300),
                                    gr.outputs.Label(type = "label", label="Predictions")],
                        title="XAI-Class Activation Map",
                        examples=examples,
                        description=description,
                        article=article,
                        live=True).launch()

