"""
Derived from https://github.com/ktatar/latent-timbre-synthesis by Tatar et. al., paper link: https://arxiv.org/pdf/2006.00408.pdf

"""

from __future__ import absolute_import, division, print_function, unicode_literals

import tensorflow as tf
from tensorflow.keras import layers
tf.keras.backend.clear_session()

import random
import numpy as np

import os, sys, argparse, time
from pathlib import Path

import librosa
import configparser
import random
import json
import matplotlib.pyplot as plt
import soundfile as sf
import cv2

parser = argparse.ArgumentParser()
parser.add_argument('--config', type=str, default ='./configs.ini' , help='path to the config file')
args = parser.parse_args()

config_path = args.config
config = configparser.ConfigParser(allow_no_value=True)
try: 
    config.read(config_path)
except FileNotFoundError:
    print('Config File Not Found at {}'.format(config_path))
    sys.exit()

#Configs 
sample_rate = config['audio'].getint('sample_rate')
hop_length = config['audio'].getint('hop_length')
bins_per_octave = config['audio'].getint('bins_per_octave')
num_octaves = config['audio'].getint('num_octaves')
n_bins = int(num_octaves * bins_per_octave)
n_iter = config['audio'].getint('n_iter')

batch_size = 32

AUTOTUNE = tf.data.experimental.AUTOTUNE

workdir2 = Path(os.getcwd())

my_examples_folder_audio_recons = workdir2.joinpath('./SSRL_Media/Designed_Audio/')

class Sampling(layers.Layer):
    """Uses (z_mean, z_log_var) to sample z, the vector encoding a digit."""

    def call(self, inputs):
        z_mean, z_log_var = inputs
        batch = tf.shape(z_mean)[0]
        dim = tf.shape(z_mean)[1]
        epsilon = tf.keras.backend.random_normal(shape=(batch, dim))
        return z_mean + tf.exp(0.5 * z_log_var) * epsilon

def synthesize_audio(np_image):
    np_image = np.rot90(np_image, 3) # to adjust the dimensions

    # *********************** LOAD MODEL **********************

    with tf.keras.utils.CustomObjectScope({'Sampling': Sampling}):
        vae = tf.keras.models.load_model('./model.hp5')
        vae.summary()

    # Encoder
    encoder_audio = tf.keras.Model(inputs = vae.get_layer("encoder_input_audio").input, outputs = [vae.get_layer("z_mean_audio").output, vae.get_layer("z_log_var_audio").output], name='encoder_audio')
    encoder_audio.summary()

    encoder_visual_score = tf.keras.Model(inputs = vae.get_layer("encoder_input_visual_score").input, outputs = [vae.get_layer("z_mean_visual_score").output, vae.get_layer("z_log_var_visual_score").output], name='encoder_visual_score')
    encoder_visual_score.summary()

    # Decoder
    decoder_audio = tf.keras.Model(inputs = vae.get_layer('decoder_audio').input, outputs = vae.get_layer('decoder_audio').output, name='decoder_audio')
    decoder_audio.summary()

    decoder_visual_score = tf.keras.Model(inputs = vae.get_layer('decoder_visual_score').input, outputs = vae.get_layer('decoder_visual_score').output, name='decoder_visual_score')
    decoder_visual_score.summary()

    # Generate examples 
    my_array = np_image / 255.0 # To scale pixel values
    
    user_input = tf.data.Dataset.from_tensor_slices(my_array).batch(batch_size).prefetch(AUTOTUNE)
    
    output_audio = tf.constant(0., dtype='float32', shape=(1,n_bins))
    
    for step, x_batch_train in enumerate(user_input):

        reconstructed_whole = vae([tf.random.uniform(shape=(1, 384)), x_batch_train]) # random uniform is for the other modality (audio)
        visual_score_z_mean_var = encoder_visual_score(x_batch_train, training = False)
        visual_score_z = Sampling()((visual_score_z_mean_var[0], visual_score_z_mean_var[1]))
        transferred_audio = decoder_audio(visual_score_z, training = False)
        output_audio = tf.concat([output_audio, transferred_audio], 0)
        
    output_np = np.transpose(output_audio.numpy())
    output_inv_32 = librosa.griffinlim_cqt(output_np[1:], sr=sample_rate, n_iter=n_iter, hop_length=hop_length, bins_per_octave=bins_per_octave, dtype=np.float32)
    sf.write(my_examples_folder_audio_recons.joinpath('generated_audio.wav'), output_inv_32, sample_rate)