import numpy as np
import torch
from pytorch_lightning.loops import Loop

from src.dataset import DATASET_REGISTRY
from src.dataset.ray_utils import denormalize_vgg, normalize_vgg
from src.loop.utils import N_to_reso, cal_n_samples
from src.model import MODEL_REGISTRY
from src.sampler.simple_sampler import SimpleSampler, InfiniteSamplerWrapper
import torch.nn.functional as TF

from src.style_module.style_module import cal_mse_content_loss, cal_adain_style_loss


class StyleTrainingLoop(Loop):
    def __init__(self, epoch, cfg, renderer):
        super().__init__()
        self.cfg = cfg
        self.model = MODEL_REGISTRY.get(self.cfg["model"]["name"])(cfg)

        self.dataloader = DATASET_REGISTRY.get(self.cfg["dataset"]["name"])(
            **self.cfg["dataset"]["train"]["params"],
        )
        self.renderer = renderer
        self.optimizer = None
        self.training_sampler = None
        self.frame_sampler = None
        self.style_dataloader = None
        self.iteration = 0
        self.epoch = epoch
        self.device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
        self.init_loop()
        self.init_optimizer()

    def init_loop(self):
        self.white_bg = self.dataloader.white_bg
        self.near_far = self.dataloader.near_far
        self.h_rays, self.w_rays = self.dataloader.img_wh[1], self.dataloader.img_wh[0]

        self.step_ratio = self.cfg["sampler"]["params"]["step_ratio"]
        self.batch_size = self.cfg["sampler"]["params"]["batch_size"]
        self.patch_size = self.cfg["sampler"]["params"]["patch_size"]
        self.chunk_size = self.cfg["sampler"]["params"]["chunk_size"]

        self.aabb = self.dataloader.scene_bbox.to(self.device)
        reso_cur = N_to_reso(self.cfg["sampler"]["params"]["N_voxel_init"], self.aabb)
        self.nSamples = min(int(self.cfg["sampler"]["params"]["n_samples"]), cal_n_samples(reso_cur, self.step_ratio))

        torch.cuda.empty_cache()
        self.dataloader.prepare_feature_data(self.model.tensorf.encoder)
        self.allrays, self.allfeatures = self.dataloader.all_rays, self.dataloader.all_features
        self.allrays_stack, self.allrgbs_stack = self.dataloader.all_rays_stack, self.dataloader.all_rgbs_stack

        if not self.model.ndc_ray:
            self.allrays, self.allfeatures = self.model.tensorf.filtering_rays(self.allrays, self.allfeatures,
                                                                               bbox_only=True)

        self.training_sampler = SimpleSampler(self.allrays.shape[0], self.batch_size)
        self.frame_sampler = iter(InfiniteSamplerWrapper(self.allrays_stack.size(0)))  # every next(sampler) returns a frame index

        self.style_dataloader = DATASET_REGISTRY.get(self.cfg["style_dataset"]["name"])(
            datadir=self.cfg["style_dataset"]["train"]["params"]["datadir"],
            batch_size=self.cfg["style_dataset"]["train"]["params"]["batch_size"],
            sampler=InfiniteSamplerWrapper,
            image_side_length=self.cfg["style_dataset"]["train"]["params"]["image_side_length"],
            num_workers=self.cfg["style_dataset"]["train"]["params"]["num_workers"],
        )

    def init_optimizer(self):
        grad_vars = self.model.tensorf.get_optparam_groups_feature_mod(self.cfg["optimizer"]["lr_init"], self.cfg["optimizer"]["lr_basis"])

        if self.cfg["optimizer"]["lr_decay_iters"] > 0:
            self.lr_factor = self.cfg["optimizer"]["lr_decay_target_ratio"] ** (1 / self.cfg["optimizer"]["lr_decay_iters"])
        else:
            self.lr_factor = self.cfg["optimizer"]["lr_decay_target_ratio"] ** (1 / self.cfg["trainer"]["n_iters"])

        print("lr decay", self.cfg["optimizer"]["lr_decay_target_ratio"], self.cfg["optimizer"]["lr_decay_iters"])

        self.optimizer = torch.optim.Adam(grad_vars, betas=(0.9, 0.99))

    def advance(self):
        # get style_img, this style_img has NOT been normalized according to the pretrained VGGmodel
        style_img = next(self.style_dataloader)[0].to(self.device)

        # randomly sample patch_size*patch_size patch from given frame
        frame_idx = next(self.frame_sampler)
        start_h = np.random.randint(0, self.h_rays - self.patch_size + 1)
        start_w = np.random.randint(0, self.w_rays - self.patch_size + 1)
        if self.white_bg:
            # move random sampled patches into center
            mid_h, mid_w = (self.h_rays - self.patch_size + 1) / 2, (self.w_rays - self.patch_size + 1) / 2
            if mid_h - start_h >= 1:
                start_h += np.random.randint(0, mid_h - start_h)
            elif mid_h - start_h <= -1:
                start_h += np.random.randint(mid_h - start_h, 0)
            if mid_w - start_w >= 1:
                start_w += np.random.randint(0, mid_w - start_w)
            elif mid_w - start_w <= -1:
                start_w += np.random.randint(mid_w - start_w, 0)

        rays_train = self.allrays_stack[frame_idx, start_h:start_h + self.patch_size, start_w:start_w + self.patch_size, :] \
            .reshape(-1, 6).to(self.device)
        # [patch*patch, 6]

        rgbs_train = self.allrgbs_stack[frame_idx, start_h:(start_h + self.patch_size),
                     start_w:(start_w + self.patch_size), :].to(self.device)
        # [patch, patch, 3]

        feature_map, acc_map, style_feature = self.renderer(rays_train, self.model.tensorf, chunk=self.chunk_size,
                                                       N_samples=self.nSamples, white_bg=self.white_bg,
                                                       ndc_ray=self.model.ndc_ray, render_feature=True, style_img=style_img,
                                                       device=self.device, is_train=True)

        feature_map = feature_map.reshape(self.patch_size, self.patch_size, 256)[None, ...].permute(0, 3, 1, 2)
        rgb_map = self.model.tensorf.decoder(feature_map)

        # feature_map is trained with normalized rgb maps, so here we don't normalize the rgb map again.
        rgbs_train = normalize_vgg(rgbs_train[None, ...].permute(0, 3, 1, 2))

        out_image_feature = self.model.tensorf.encoder(rgb_map)
        content_feature = self.model.tensorf.encoder(rgbs_train)

        if self.white_bg:
            mask = acc_map.reshape(self.patch_size, self.patch_size, 1)[None, ...].permute(0, 3, 1, 2)
            # if not (mask > 0.5).any():
            #     continue

            # content loss
            _mask = TF.interpolate(mask, size=content_feature.relu4_1.size()[-2:], mode='bilinear').ge(1e-5)
            content_loss = cal_mse_content_loss(torch.masked_select(content_feature.relu4_1, _mask),
                                                torch.masked_select(out_image_feature.relu4_1, _mask))
            # style loss
            style_loss = 0.
            for style_feature, image_feature in zip(style_feature, out_image_feature):
                _mask = TF.interpolate(mask, size=image_feature.size()[-2:], mode='bilinear').ge(1e-5)
                C = image_feature.size()[1]
                masked_img_feature = torch.masked_select(image_feature, _mask).reshape(1, C, -1)
                style_loss += cal_adain_style_loss(style_feature, masked_img_feature)

            content_loss *= self.cfg["style_config"]["content_weight"]
            style_loss *= self.cfg["style_config"]["style_weight"]
        else:
            # content loss
            content_loss = cal_mse_content_loss(content_feature.relu4_1, out_image_feature.relu4_1)
            # style loss
            style_loss = 0.
            for style_feature, image_feature in zip(style_feature, out_image_feature):
                style_loss += cal_adain_style_loss(style_feature, image_feature)

            content_loss *= self.cfg["style_config"]["content_weight"]
            style_loss *= self.cfg["style_config"]["style_weight"]

        feature_tv_loss = self.model.tensorf.tvreg(feature_map) * self.cfg["style_config"]["featuremap_tv_weight"]
        image_tv_loss = self.model.tensorf.tvreg(denormalize_vgg(rgb_map)) * self.cfg["style_config"]["image_tv_weight"]

        total_loss = content_loss + style_loss + feature_tv_loss + image_tv_loss

        self.iteration += 1

        self.optimizer.zero_grad()
        total_loss.backward()
        self.optimizer.step()

        for param_group in self.optimizer.param_groups:
                param_group['lr'] = param_group['lr'] * self.lr_factor





