import numpy as np
import torch
from pytorch_lightning.loops import Loop

from src.dataset import DATASET_REGISTRY
from src.dataset.ray_utils import denormalize_vgg, normalize_vgg
from src.loop.utils import N_to_reso, cal_n_samples
from src.model import MODEL_REGISTRY
from src.sampler.simple_sampler import SimpleSampler, InfiniteSamplerWrapper
import torch.nn.functional as TF


class FeatureTrainingLoop(Loop):
    def __init__(self, epoch, cfg, renderer):
        super().__init__()
        self.cfg = cfg
        self.model = MODEL_REGISTRY.get(self.cfg["model"]["name"])(cfg)

        self.dataloader = DATASET_REGISTRY.get(self.cfg["dataset"]["name"])(
                **self.cfg["dataset"]["train"]["params"],
            )
        self.renderer = renderer
        self.optimizer = None
        self.training_sampler = None
        self.frame_sampler = None
        self.iteration = 0
        self.epoch = epoch
        self.device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
        self.init_loop()
        self.init_optimizer()

    def init_loop(self):
        self.white_bg = self.dataloader.white_bg
        self.near_far = self.dataloader.near_far
        self.h_rays, self.w_rays = self.dataloader.img_wh[1], self.dataloader.img_wh[0]

        self.step_ratio = self.cfg["sampler"]["params"]["step_ratio"]
        self.batch_size = self.cfg["sampler"]["params"]["batch_size"]
        self.patch_size = self.cfg["sampler"]["params"]["patch_size"]
        self.chunk_size = self.cfg["sampler"]["params"]["chunk_size"]

        self.aabb = self.dataloader.scene_bbox.to(self.device)
        reso_cur = N_to_reso(self.cfg["sampler"]["params"]["N_voxel_init"], self.aabb)
        self.nSamples = min(int(self.cfg["sampler"]["params"]["n_samples"]), cal_n_samples(reso_cur, self.step_ratio))

        torch.cuda.empty_cache()
        self.dataloader.prepare_feature_data(self.model.tensorf.encoder)
        self.allrays, self.allfeatures = self.dataloader.all_rays, self.dataloader.all_features
        self.allrays_stack, self.allrgbs_stack = self.dataloader.all_rays_stack, self.dataloader.all_rgbs_stack

        if not self.model.ndc_ray:
            self.allrays, self.allfeatures = self.model.tensorf.filtering_rays(self.allrays, self.allfeatures, bbox_only=True)

        self.training_sampler = SimpleSampler(self.allrays.shape[0], self.batch_size)
        self.frame_sampler = iter(InfiniteSamplerWrapper(self.allrays_stack.size(0)))  # every next(sampler) returns a frame index

    def init_optimizer(self):
        grad_vars = self.model.tensorf.get_optparam_groups_feature_mod(self.cfg["optimizer"]["lr_init"], self.cfg["optimizer"]["lr_basis"])

        if self.cfg["optimizer"]["lr_decay_iters"] > 0:
            self.lr_factor = self.cfg["optimizer"]["lr_decay_target_ratio"] ** (1 / self.cfg["optimizer"]["lr_decay_iters"])
        else:
            self.lr_factor = self.cfg["optimizer"]["lr_decay_target_ratio"] ** (1 / self.cfg["trainer"]["n_iters"])

        print("lr decay", self.cfg["optimizer"]["lr_decay_target_ratio"], self.cfg["optimizer"]["lr_decay_iters"])

        self.optimizer = torch.optim.Adam(grad_vars, betas=(0.9, 0.99))

    @property
    def done(self):
        """Advance from one iteration to the next."""
        return self.epoch < self.iteration

    def reset(self):
        """Advance from one iteration to the next."""

    def advance(self):
        """Advance from one iteration to the next."""
        feature_loss, pixel_loss = 0., 0.
        if self.iteration % 2 == 0:
            ray_idx = self.training_sampler.nextids()
            rays_train, features_train = self.allrays[ray_idx], self.allfeatures[ray_idx].to(self.device)

            feature_map, _ = self.renderer(rays_train, self.model.tensorf, chunk=self.chunk_size, N_samples=self.nSamples, white_bg=self.white_bg,
                                      ndc_ray=self.model.ndc_ray, render_feature=True, device=self.device, is_train=True)

            feature_loss = torch.mean((feature_map - features_train) ** 2)
        else:
            frame_idx = next(self.frame_sampler)
            start_h = np.random.randint(0, self.h_rays - self.patch_size  + 1)
            start_w = np.random.randint(0, self.w_rays - self.patch_size  + 1)
            if self.white_bg:
                # move random sampled patches into center
                mid_h, mid_w = (self.h_rays - self.patch_size + 1) / 2, (self.w_rays - self.patch_size + 1) / 2
                if mid_h - start_h >= 1:
                    start_h += np.random.randint(0, mid_h - start_h)
                elif mid_h - start_h <= -1:
                    start_h += np.random.randint(mid_h - start_h, 0)
                if mid_w - start_w >= 1:
                    start_w += np.random.randint(0, mid_w - start_w)
                elif mid_w - start_w <= -1:
                    start_w += np.random.randint(mid_w - start_w, 0)

            rays_train = self.allrays_stack[frame_idx, start_h:start_h + self.patch_size ,
                         start_w:start_w + self.patch_size , :].reshape(-1, 6).to(self.device)
            # [patch*patch, 6]

            rgbs_train = self.allrgbs_stack[frame_idx, start_h:(start_h + self.patch_size ),
                         start_w:(start_w + self.patch_size ), :].to(self.device)
            # [patch, patch, 3]

            feature_map, _ = self.renderer(rays_train, self.model.tensorf, chunk=self.chunk_size, N_samples=self.nSamples, white_bg=self.white_bg,
                                      ndc_ray=self.model.ndc_ray, render_feature=True, device=self.device, is_train=True)

            feature_map = feature_map.reshape(self.patch_size , self.patch_size , 256)[None, ...].permute(0, 3, 1, 2)
            recon_rgb = self.model.tensorf.decoder(feature_map)

            rgbs_train = rgbs_train[None, ...].permute(0, 3, 1, 2)
            img_enc = self.model.tensorf.encoder(normalize_vgg(rgbs_train))
            recon_rgb_enc = self.model.tensorf.encoder(recon_rgb)

            feature_loss = (TF.mse_loss(recon_rgb_enc.relu4_1, img_enc.relu4_1) +
                            TF.mse_loss(recon_rgb_enc.relu3_1, img_enc.relu3_1)) / 10

            recon_rgb = denormalize_vgg(recon_rgb)

            pixel_loss = torch.mean((recon_rgb - rgbs_train) ** 2)

        total_loss = pixel_loss + feature_loss

        # loss
        # NOTE: Calculate feature TV loss rather than appearence TV loss
        if self.model.TV_weight_feature > 0:
            self.model.TV_weight_feature *= self.lr_factor
            loss_tv = self.model.tensorf.TV_loss_feature(self.model.tvreg) * self.model.TV_weight_feature
            total_loss = total_loss + loss_tv

        self.iteration += 1

        self.optimizer.zero_grad()
        total_loss.backward()
        self.optimizer.step()



