import time
import gradio as gr
from load_checkpoint import load_checkpoint
import commons
from inference import SynthesizerInf
from text import text_to_sequence
from torch import no_grad, LongTensor
import torch
from text.symbols import symbols
from hparams import get_hparams_from_file

hps_ms = get_hparams_from_file(r'./9nine.json')
device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

net_g_ms = SynthesizerInf(
    len(symbols),
    hps_ms.data.filter_length // 2 + 1,
    hps_ms.train.segment_size // hps_ms.data.hop_length,
    n_speakers=hps_ms.data.n_speakers,
    **hps_ms.model).to(device)

_ = net_g_ms.eval()

model, optimizer, learning_rate, epochs = load_checkpoint(r'./9nine_G_196000.pth', net_g_ms, None)

def get_text(text, hps):
    text_norm = text_to_sequence(text, hps.data.text_cleaners)
    if hps.data.add_blank:
        text_norm = commons.intersperse(text_norm, 0)
    text_norm = torch.LongTensor(text_norm)
    return text_norm

def vits(text, speaker_id, noise_scale, noise_scale_w, length_scale):
    start = time.perf_counter()
    if not len(text):
        return "输入文本不能为空！", None, None
    text = text.replace('\n', ' ').replace('\r', '').replace(" ", "")
    if len(text) > 500:
        return f"输入文字过长！{len(text)}>100", None, None

    stn_tst = get_text(text, hps_ms)

    with no_grad():
        x_tst = stn_tst.unsqueeze(0)
        x_tst_lengths = LongTensor([stn_tst.size(0)])
        speaker_id = LongTensor([speaker_id])
        audio = net_g_ms.forward(x_tst, x_tst_lengths, sid=speaker_id, noise_scale=noise_scale, noise_scale_w=noise_scale_w,
                               length_scale=length_scale)[0][0, 0].data.cpu().float().numpy()

    return "生成成功!", (22050, audio), f"生成耗时 {round(time.perf_counter()-start, 2)} s"


download_audio_js = """
() =>{{
    let root = document.querySelector("body > gradio-app");
    if (root.shadowRoot != null)
        root = root.shadowRoot;
    let audio = root.querySelector("#tts-audio").querySelector("audio");
    let text = root.querySelector("#input-text").querySelector("textarea");
    if (audio == undefined)
        return;
    text = text.value;
    if (text == undefined)
        text = Math.floor(Math.random()*100000000);
    audio = audio.src;
    let oA = document.createElement("a");
    oA.download = text.substr(0, 20)+'.wav';
    oA.href = audio;
    document.body.appendChild(oA);
    oA.click();
    oA.remove();
}}
"""

if __name__ == '__main__':
    with gr.Blocks() as app:
        gr.Markdown(
            "# <center> 9Nine - VITS\n"
            '<div align="center"><a><font color="#dd0000">结果有随机性，语调可能很奇怪，可多次生成取最佳效果</font></a></div>'
            '<div align="center"><a><font color="#dd0000">标点符号会影响生成的结果</font></a></div>'
        )

        with gr.Column():
            input_text = gr.Textbox(label="Text (100 words limitation)", lines=5, 
            value="そんなわけないじゃない。どうしてこうなるだろう。始めて好きな人ができた。一生ものの友达ができた。嬉しいことが二つ重なて。その二つの嬉しさがまたたくさんの嬉しさをつれて来てくれて。梦のように幸せの时间を手に入れたはずなのに。なのにどうして、こうなちょうだろう。", 
            elem_id=f"input-text")
            btn = gr.Button(value="Submit")

            sid = gr.Dropdown(label="Speaker", choices=["0", "1", "2", "3", "4"], type="index", value="1")

            with gr.Row():
                ns = gr.Slider(label="noise_scale(控制感情变化程度)", minimum=0.1, maximum=1.0, step=0.1, value=0.6, interactive=True)
                nsw = gr.Slider(label="noise_scale_w(控制音素发音长度)", minimum=0.1, maximum=1.0, step=0.1, value=0.668, interactive=True)
                ls = gr.Slider(label="length_scale(控制整体语速)", minimum=0.1, maximum=2.0, step=0.1, value=1.2, interactive=True)

        with gr.Column():
            o1 = gr.Textbox(label="Output Message")
            o2 = gr.Audio(label="Output Audio", elem_id=f"tts-audio")
            o3 = gr.Textbox(label="Extra Info")
            download = gr.Button("Download Audio")

        btn.click(vits, inputs=[input_text, sid, ns, nsw, ls], outputs=[o1, o2, o3], api_name="generate")
        download.click(None, [], [], _js=download_audio_js.format())

    app.queue(concurrency_count=1).launch()
