from datetime import datetime, timedelta
from decimal import Decimal
import gradio as gr
import matplotlib.pyplot as plt
from pathlib import Path
import pickle
import os
import time
import torch
import torchvision.transforms as T

from ocean_lib.ocean.ocean import Ocean
from ocean_lib.config import Config
from ocean_lib.models.compute_input import ComputeInput
from ocean_lib.web3_internal.wallet import Wallet



def compute(private_key):

    start_time = time.time()

    network = 'mumbai'
    config = Config(f'configs/config-{network}.ini')
    network_url = Path(config.network_url[8:])
    network_url = network_url.parents[0] /  os.getenv('INFURA_KEY')
    config.set('eth-network', 'network', "https://" + str(network_url))
    ocean = Ocean(config)

    wallet = Wallet(ocean.web3, str(private_key).strip(), config.block_confirmations, config.transaction_timeout)

    address = wallet.address

    data_did = {
        "rinkeby": "did:op:064abd2c7f8d5c3cacdbf43a687194d50008889130dbc4403d4b973797da7081",
        "mumbai": "did:op:2def598c3ab3732dec5db8f33af028429a547b101cf6e7ec4ac9eba547a5507d"
    }

    data_asset = ocean.assets.resolve(data_did[config.network_name])
    data_token = ocean.get_datatoken(data_asset.datatokens[0]['address'])
    data_did = data_asset.did

    assert data_token.balanceOf(wallet.address) > 0, "need to buy data token"
    
    algo_did = {
        "rinkeby": "did:op:b6df860d4db1405293768985600db2de317d8f0035aa1805c0724ae678f23477",
        "mumbai": "did:op:0a618a6c19ea44a189598f71aaefc98a516592cf90c64fd1f81f3f840f7b5b92"
    }

    algo_asset = ocean.assets.resolve(algo_did[config.network_name])
    algo_token = ocean.get_datatoken(algo_asset.datatokens[0]['address'])
    algo_did = algo_asset.did

    assert algo_token.balanceOf(wallet.address) > 0, "need to buy data token"
    
    compute_service = data_asset.services[0]
    algo_service = algo_asset.services[0]
    free_c2d_env = ocean.compute.get_free_c2d_environment(compute_service.service_endpoint)

    data_compute_input = ComputeInput(data_asset, compute_service)
    algo_compute_input = ComputeInput(algo_asset, algo_service)

    if config.network_name == 'rinkeby' or config.network_name == 'mainnet':
        assert alice_wallet.web3.eth.get_balance(alice_wallet.address) > 0, "need ETH"
    # elif config.network_name == 'mumbai' or config.network_name == 'polygon':
    #     assert ocean.MATIC_token.balanceOf(alice_wallet.address) > 0, "need MATIC"
    
    # Pay for dataset and algo for 1 day
    datasets, algorithm = ocean.assets.pay_for_compute_service(
        datasets=[data_compute_input],
        algorithm_data=algo_compute_input,
        consume_market_order_fee_address=wallet.address,
        wallet=wallet,
        compute_environment=free_c2d_env["id"],
        valid_until=int((datetime.utcnow() + timedelta(days=1)).timestamp()),
        consumer_address=free_c2d_env["consumerAddress"],
    )
    assert datasets, "pay for dataset unsuccessful"
    assert algorithm, "pay for algorithm unsuccessful"

    # Start compute job
    job_id = ocean.compute.start(
        consumer_wallet=wallet,
        dataset=datasets[0],
        compute_environment=free_c2d_env["id"],
        algorithm=algorithm,
    )

    # Wait until job is done
    succeeded = False
    for _ in range(0, 200):
        status = ocean.compute.status(data_asset, compute_service, job_id, wallet)
        print('===========', status['statusText'])
        
        if status.get("dateFinished") and Decimal(status["dateFinished"]) > 0:
            succeeded = True
            break
        time.sleep(5)

    print('hii')

    # Retrieve algorithm output and log files
    result = ocean.compute.compute_job_result_logs(
        data_asset, compute_service, job_id, wallet
    )[0]

    print('===================0', result)

    tensor = pickle.loads(result) 

    print('===================1', tensor)

    img = T.ToPILImage()(tensor.permute(2,0,1))
    
    print('===================2', img)

    elapsed_time = time.time() - start_time
        
    print('===================3', elapsed_time)

    return address, img

description = (
    "This demo serves a generative model from the Ocean marketplace. "
)

article = (
    "<p style='text-align: center'>"
    "<a href='https://metahub.algovera.ai/' target='_blank'>Algovera Metahub</a> | "
    "<a href='https://market.oceanprotocol.com/' target='_blank'>Ocean Marketplace</a> | "
    "<a href='https://docs.algovera.ai/blog/2022/04/05/Tutorial%20for%20using%20token-gated%20apps%20on%20HuggingFace' target='_blank'>How-to-use Tutorial</a> | "
    "<a href='https://www.algovera.ai' target='_blank'>Algovera Website</a>"
    "</p>"
)

interface = gr.Interface(
    compute,
    [
        gr.inputs.Textbox(label="Private Key"),
    ],
    [
        gr.outputs.Textbox(label="Public Key"),
        gr.outputs.Image(label="Output Image")
    ],
    title="Generative Model from the Ocean marketplace",
    description=description,
    article=article,
)

interface.launch()