from turtle import width
import pandas as pd
import numpy as np
from random import normalvariate, random
import plotly.express as px

from cadCAD.configuration.utils import config_sim
from cadCAD.configuration import Experiment
from cadCAD.engine import ExecutionContext, Executor
from cadCAD import configs
import streamlit as st


# Additional dependencies

# For analytics
import numpy as np
# For visualization
import plotly.express as px
pd.options.plotting.backend = "plotly"

st.header('Algovera Active/Passive Squad Model')

def p_active(params, substep, state_history, previous_state):
    if (previous_state['timestep'] < params['active_timesteps']) and (previous_state['funding_pool'] > params['funding_round']):
        to_squad = params['funding_round']
        return {'update_squad_funding': to_squad,
                'update_funding_pool': -to_squad}
    return {'update_squad_funding': 0,
            'update_funding_pool': 0}

def p_init_algo(params, substep, state_history, previous_state):
    if (previous_state['timestep'] > params['active_timesteps']) and (previous_state['liquidity'] == 0):
        to_liquidity = params['init_liquidity']
        mint_tokens = to_liquidity * params['token_ratio']
        token_price = to_liquidity / mint_tokens
        return {'update_liquidity_pool': to_liquidity,
                'update_squad_funding': -to_liquidity,
                'update_token_supply': mint_tokens,
                'update_token_price': token_price}
    return {'update_liquidity_pool': 0,
            'update_squad_funding': 0,
            'update_token_supply': 0,
            'update_token_price': 0}

def p_passive(params, substep, state_history, previous_state):
    if (previous_state['timestep'] >= params['active_timesteps']):
        if (previous_state['token_supply'] > 0):
            token_buy = 1
            price = previous_state['liquidity'] / previous_state['token_supply']
            if (previous_state['consumer_LT'] > price) and ((previous_state['token_supply'] - token_buy) > 0):
                updated_price = (previous_state['liquidity'] + price) / (previous_state['token_supply'] - token_buy)
                fee = price * params['tx_fee']
                    
                return {'update_token_supply': -token_buy,
                        'update_liquidity_pool': price,
                        'update_consumer_tokens': token_buy,
                        'update_consumer_LT': -(price+fee),
                        'update_token_price': updated_price,
                        'update_squad_funding': fee}
            else:
                return {'update_token_supply': 0,
                        'update_liquidity_pool': 0,
                        'update_consumer_tokens': 0,
                        'update_consumer_LT': 0,
                        'update_token_price': previous_state['token_price'],
                        'update_squad_funding': 0}
    return {'update_token_supply': 0,
                    'update_liquidity_pool': 0,
                    'update_consumer_tokens': 0,
                    'update_consumer_LT': 0,
                    'update_token_price': 0,
                    'update_squad_funding': 0}

def s_timestep(params, substep, state_history, previous_state, policy_input):
    updated_timestep = previous_state['timestep'] + 1
    return 'timestep', updated_timestep

def s_squad(params, substep, state_history, previous_state, policy_input):
    tokens = previous_state['squad_tokens']
    updated_tokens = tokens + policy_input['update_squad_funding']
    return 'squad_tokens', updated_tokens

def s_funding_pool(params, substep, state_history, previous_state, policy_input):
    funding_pool = previous_state['funding_pool']
    updated_funding_pool = funding_pool + policy_input['update_funding_pool']
    return 'funding_pool', updated_funding_pool

def s_consumer_LT(params, substep, state_history, previous_state, policy_input):
    consumer_LT = previous_state['consumer_LT']
    updated_consumer_LT = consumer_LT + policy_input['update_consumer_LT']
    return 'consumer_LT', updated_consumer_LT

def s_consumer_tokens(params, substep, state_history, previous_state, policy_input):
    consumer_tokens = previous_state['consumer_tokens']
    updated_consumer_tokens = consumer_tokens + policy_input['update_consumer_tokens']
    return 'consumer_tokens', updated_consumer_tokens

def s_token_price(params, substep, state_history, previous_state, policy_input):
    updated_token_price = policy_input['update_token_price']
    return 'token_price', updated_token_price

def s_token_supply(params, substep, state_history, previous_state, policy_input):
    token_supply = previous_state['token_supply']
    updated_token_supply = token_supply + policy_input['update_token_supply']
    return 'token_supply', updated_token_supply

def s_liquidity_pool(params, substep, state_history, previous_state, policy_input):
    liquidity_pool = previous_state['liquidity']
    updated_liquidity_pool = liquidity_pool + policy_input['update_liquidity_pool']
    return 'liquidity', updated_liquidity_pool

st.subheader('Simulation Parameters')
st.subheader('Active Parameters')
st.write('Initial Funding Pool')
funding_pool = st.slider('Initial Funding Pool', min_value=1000, max_value=10000, value=1000, step=10)
st.write('Set the funding disbursed each active round from the funding pool')
funding_round = st.slider('Funding Round', min_value=100, max_value=1000, value=100, step=1)

st.write('Set the number of active timesteps where funding is directly sent to the squad.')
active_timesteps = st.slider('Active Timesteps', min_value=0, max_value=100, value=10, step=1)

st.subheader('Passive Parameters')
st.write('Set the initial liquidity to be supplied to the algorithm liquidity pool. Note: make sure the squad will have enough tokens!')
init_liquidity = st.slider('Initial Liquidity', min_value=0, max_value=5000, value=10, step=1)
st.write('Set the relative transaction fee to be fed back to the squad treasury during passive phase.')
tx_fee = st.slider('Transaction Fee', min_value=0., max_value=1., value=0.1, step=0.0001)
st.write('Set the number of liquidity tokens consumer has to buy algorithm tokens.')
consumer_LT = st.slider('Consumer LT', min_value=0, max_value=10000, value=100, step=1)
st.write('Set the initial ratio of liquidity tokens and algorithm tokens. This is used to mint AT and determine the price of AT in liquidity tokens.')
st.latex(r'''
\frac{P_a \cdot A}{L}=1
''')
st.write('P_a is the price of the algorithm token (in liquidity tokens), A is the supply of algorithm tokens, L is the supply of liquidity tokens.')
st.write('Initial supply is:')
st.latex(r'''
A_i = L_i \cdot r
''')
st.write('where r is the Token Ratio.')
token_ratio = st.number_input('Token Ratio', value=0.7)
timesteps = st.slider('Simulation Timesteps', min_value=10, max_value=10000, value=100, step=1)

initial_state = {
    'funding_pool': funding_pool,
    'funding_round': funding_round,
    'squad_tokens': 0,
    'active_timesteps': active_timesteps,
    'liquidity': 0,
    'init_liquidity': init_liquidity,
    'timestep': 0,
    'tx_fee': tx_fee,
    'consumer_LT': consumer_LT,
    'consumer_tokens': 0,
    'token_supply': 0,
    'token_ratio': token_ratio,
    'token_price': 0.0
}

system_params = {
    'funding_pool': [funding_pool],
    'funding_round': [funding_round],
    'squad_tokens': [0],
    'active_timesteps': [active_timesteps],
    'liquidity': [0],
    'init_liquidity': [init_liquidity],
    'timestep': [0],
    'tx_fee': [tx_fee],
    'consumer_LT': [consumer_LT],
    'consumer_tokens': [0],
    'token_supply': [0],
    'token_ratio': [token_ratio],
    'token_price': [0.0]
}

def generate_sim_config(monte_carlo_runs=1,
                   timesteps=timesteps,
                   system_params=system_params):
    sim_config = config_sim({
        'N': monte_carlo_runs, # the number of times we'll run the simulation ("Monte Carlo runs")
        'T': range(timesteps), # the number of timesteps the simulation will run for
        'M': system_params # the parameters of the system
    })

    return sim_config

def configure_experiment(initial_state,
                      partial_state_update_blocks,
                      sim_config):
    experiment = Experiment()
    experiment.append_configs(
        initial_state=initial_state,
        partial_state_update_blocks=partial_state_update_blocks,
        sim_configs=sim_config
    )

    return experiment

partial_state_update_blocks = [
    {
        'policies': {
            'p_active': p_active,
            'p_init_algo': p_init_algo,
            'p_passive': p_passive
        },
        'variables': {
            'timestep': s_timestep,
            'funding_pool': s_funding_pool,
            'squad_tokens': s_squad,
            'consumer_LT': s_consumer_LT,
            'consumer_tokens': s_consumer_tokens,
            'liquidity': s_liquidity_pool,
            'token_supply': s_token_supply,
            'token_price': s_token_price
        }
    }
]

def execute_simulation(experiment):
    exec_context = ExecutionContext()
    configs = experiment.configs
    simulation = Executor(exec_context=exec_context, configs=configs)
    raw_result, tensor_field, sessions = simulation.execute()

    return raw_result

if st.button('Run Simulation'):
    sim_config = generate_sim_config()
    experiment = configure_experiment(initial_state, partial_state_update_blocks, sim_config)
    raw_result = execute_simulation(experiment)
    df = pd.DataFrame(raw_result)
    fig1 = df.plot(kind='line', x='timestep', y=['funding_pool','squad_tokens'], width=1000)
    fig2 = df.plot(kind='line', x='timestep', y=['token_supply','token_price', 'liquidity'], width=1000)
    fig3 = df.plot(kind='line', x='timestep', y=['squad_tokens','consumer_LT'], width=1000)
    st.subheader('Results')
    st.plotly_chart(fig1)
    st.plotly_chart(fig2)
    st.plotly_chart(fig3)
