from functools import partial

import gradio as gr

import config
from modules import utils
from modules import common
from modules.models import GuanacoModel, ChaiBot


def render_playground(demo):
    # set inital states
    bot_config = utils.get_bot_config(config.DEFAULT_BOT_NAME)
    bot_state = gr.State(bot_config)
    convo_state = common.get_convo_state(bot_config)

    # render widgets
    render_header()
    common.render_section_separator("Set up")
    model_tag = common.render_model_selector()
    bot_profile, bot_selector = common.render_bot_profile(bot_config)
    bot_config_text = common.render_bot_config(bot_config)

    common.render_section_separator("Chat")
    dialog = render_dialog(bot_config)

    # set default model state according to database
    model_state = common.get_model_state(config.DEFAULT_MODEL)

    # render submit buttons and parameter sliders
    msg, send, regenerate, clear = common.render_chat_buttons()

    # set callbacks
    bot_selector.change(
        _reload_bot,
        [bot_selector, bot_profile],
        [bot_profile, convo_state, dialog, bot_state, bot_config_text],
        queue=False
    )

    model_tag.change(
        _clear_chat,
        [dialog, bot_state],
        [dialog],
        queue=False
    )
    send.click(
        _respond,
        [msg, convo_state, dialog, model_state],
        [msg, dialog],
        queue=False
    )
    msg.submit(
        _respond,
        [msg, convo_state, dialog, model_state],
        [msg, dialog],
        queue=False
    )
    regenerate.click(
        _regenerate_response,
        [convo_state, dialog, model_state],
        [dialog],
        queue=False
    )
    clear.click(
        _clear_chat,
        [dialog, bot_state],
        [dialog],
        queue=False
    )


def _update_model_parameter_slider(slider, params_state, label):
    params_state.update({label: slider})
    return params_state


def render_header():
    gr.Markdown("""
    # Playground
    """)


def render_dialog(bot_config):
    first_message = (None, bot_config["firstMessage"])
    dialog = gr.Chatbot([first_message])
    return dialog


def _reload_bot(bot_selector, bot_profile):
    bot_selector = bot_selector or config.DEFAULT_BOT_NAME
    bot_config = utils.get_bot_config(bot_selector)
    bot_profile = utils.get_bot_picture_html(bot_config)
    convo_state = ChaiBot(bot_config)
    bot_config_text = f"# Memory\n{bot_config.get('memory', '')}\n# Prompt\n{bot_config.get('prompt', '')}"
    dialog_st = [(None, bot_config["firstMessage"])]
    return bot_profile, convo_state, dialog_st, bot_config, bot_config_text


def _respond(user_message, chaibot, chat_history, model):
    chaibot.add_user_message(user_message)
    bot_response = model.generate_response(chaibot)
    chaibot.add_bot_message(bot_response)
    chat_history.append(
        (user_message, bot_response)
    )
    return "", chat_history


def _clear_chat(chat_history, bot_state):
    chat_history = [(None, bot_state["firstMessage"])]
    return chat_history


def _regenerate_response(chaibot, chat_history, model):
    chaibot.messages.pop()
    chat_history.pop()
    user_message = chaibot.messages[-1][-1]
    bot_response = model.generate_response(chaibot)
    chaibot.add_bot_message(bot_response)
    chat_history.append(
        (user_message, bot_response)
    )
    return chat_history


def _get_model(model_tag):
    model = GuanacoModel(model_tag)
    return model


def _parse_model_parameters_from_bot_id(model_tag):
    model = _get_model(model_tag)
    out = [
        model.config.generation_params["temperature"],
        model.config.generation_params["repetition_penalty"],
        model.config.generation_params["max_new_tokens"],
        model.config.generation_params["top_k"],
        model.config.generation_params["top_p"],
        model
    ]
    return out
