import requests
import gradio as gr

from config import GUANACO_DEVELOPER_KEY, MODELS


class ModelConfig():
    def __init__(self, config):
        self.name = config['name']
        self.endpoint = config['endpoint']
        self.generation_params = config.get('params', {})
        self.author_id = config.get('author-id')


class ChaiBot():
    def __init__(self, bot_config):
        self.messages = []
        self.config = bot_config
        self.bot_label = bot_config.get("botLabel", "Character")
        self.user_label = bot_config.get("userLabel", "User")
        self.add_bot_message(bot_config.get("firstMessage"))

    def add_user_message(self, message):
        self.messages.append((self.user_label, message.strip()))

    def add_bot_message(self, message):
        self.messages.append((self.bot_label, message.strip()))

    def get_conversation(self):
        conversation = []
        for label, value in self.messages:
            role_type = "user" if label == self.user_label else "bot"
            message = {
                "from": label,
                "value": value,
                "role_type": role_type
            }
            conversation.append(message)
        return conversation


class BaseModel:
    def __init__(self, model_config):
        self.config = model_config

    def generate_response(self, chaibot):
        raise NotImplemented


class GuanacoModel(BaseModel):
    def generate_response(self, chaibot):
        model_inputs = self._get_model_input(chaibot)
        return self._get_response(model_inputs)

    def _get_model_input(self, chaibot):
        model_inputs = {
            "bot_name": chaibot.bot_label,
            "memory": chaibot.config.get('memory', ""),
            "prompt": chaibot.config.get('prompt', ""),
            "chat_history": [{"sender": sender, "message": message} for sender, message in chaibot.messages],
            "user_name": "You"
        }
        return model_inputs

    def _get_response(self, inputs):
        headers = {"Authorization": f"Bearer {GUANACO_DEVELOPER_KEY}"}
        model_id = MODELS[self.config]
        url = f'https://guanaco-submitter.chai-research.com/models/{model_id}/chat'
        try:
            response = requests.post(url=url, json=inputs, headers=headers, timeout=20)
        except requests.ReadTimeout:
            raise gr.Error("Generating response took too long, please try again in new conversation.")
        return response.json()["model_output"]
