# -*- coding:utf-8 -*-
import os
import logging
import sys

import gradio as gr

from modules import config
from modules.config import *
from modules.utils import *
from modules.presets import *
from modules.overwrites import *
from modules.models.models import get_model

gr.Chatbot._postprocess_chat_messages = postprocess_chat_messages
gr.Chatbot.postprocess = postprocess

PromptHelper.compact_text_chunks = compact_text_chunks

with open("assets/custom.css", "r", encoding="utf-8") as f:
    customCSS = f.read()


def create_new_model(current_system_prompt="You are a helpful assistant."):
    return get_model(model_name=MODELS[DEFAULT_MODEL], access_key=my_api_key, system_prompt=current_system_prompt)[0]


def get_system_prompt_content_by_role_name(role_name: str) -> str:
    template_name_list = get_template_names(plain=True)
    template_role_prompt_dict = load_template(template_name_list[0], mode=2)    # [act:prompt]
    prompt_content = template_role_prompt_dict[role_name]
    return prompt_content


def get_role_name_by_id(prompt_id) -> str:
    template_name_list = get_template_names(plain=True)
    template_id_role_dict = load_template(template_name_list[0], mode=3)    # [id:act]）
    role_name = template_id_role_dict[prompt_id]
    return role_name


def get_user_key_by_user_name(user_login_name: str) -> str:
    user_key_pairs_dict = {row[0]: row[1] for row in user_key_pairs_list}

    return user_key_pairs_dict[user_login_name]


with gr.Blocks(css=customCSS, theme=small_and_beautiful_theme) as demo:
    # Session State
    user_name = gr.State("")
    user_question = gr.State("")
    user_topic = gr.State(i18n("未命名对话历史记录"))

    assert type(my_api_key) == str
    user_api_key = gr.State(my_api_key)
    current_model = gr.State(create_new_model())
    current_prompt_template = gr.State(load_template(get_template_names(plain=True)[0], mode=2))

    # Header
    with gr.Row():
        gr.HTML(CHUANHU_TITLE, elem_id="app_title")
        status_display = gr.Markdown(get_geoip(), elem_id="status_display")
    with gr.Row(elem_id="float_display", visible=True):
        user_info = gr.Markdown(value="getting user info...", elem_id="user_info")

    # Body
    with gr.Row().style(equal_height=True):
        # Left Panel
        with gr.Column(scale=5):
            with gr.Row():
                chatbot = gr.Chatbot(label="QYL Chat", elem_id="chuanhu_chatbot").style(height="100%")
            with gr.Row():
                with gr.Column(min_width=225, scale=12):
                    user_input = gr.Textbox(
                        elem_id="user_input_tb",
                        show_label=False, placeholder=i18n("在这里输入")
                    ).style(container=False)
                with gr.Column(min_width=42, scale=1):
                    submitBtn = gr.Button(value="", variant="primary", elem_id="submit_btn")
                    cancelBtn = gr.Button(value="", variant="secondary", visible=False, elem_id="cancel_btn")
            with gr.Row():
                emptyBtn = gr.Button(
                    i18n("🧹 新的对话"), elem_id="empty_btn"
                )
                retryBtn = gr.Button(i18n("🔄 重新生成"))
                delFirstBtn = gr.Button(i18n("🗑️ 删除最旧对话"), visible=False)
                delLastBtn = gr.Button(i18n("🗑️ 删除最新对话"), visible=False)
                with gr.Row(visible=False) as like_dislike_area:
                    with gr.Column(min_width=20, scale=1):
                        likeBtn = gr.Button(i18n("👍"))
                    with gr.Column(min_width=20, scale=1):
                        dislikeBtn = gr.Button(i18n("👎"))
        # Right Panel
        with gr.Column():
            with gr.Column(min_width=50, scale=1):
                with gr.Tab(label=i18n("对话")):
                    # with gr.Accordion(label=i18n("保存/加载对话历史记录"), open=True):
                    with gr.Accordion(label=i18n("加载对话历史记录"), open=True):
                        with gr.Column():
                            with gr.Row():
                                with gr.Column(scale=6):
                                    historyFileSelectDropdown = gr.Dropdown(
                                        label=i18n("从列表中加载对话"),
                                        choices=get_history_names(plain=True),
                                        multiselect=False
                                    ).style(container=False)
                                with gr.Column(scale=1):
                                    historyRefreshBtn = gr.Button(i18n("🔄 刷新"))
                            with gr.Row(visible=False):
                                with gr.Column(scale=6):
                                    saveFileName = gr.Textbox(
                                        show_label=True,
                                        placeholder=i18n("设置文件名: 默认为.json，可选为.md"),
                                        label=i18n("设置保存文件名"),
                                        value=i18n("对话历史记录"),
                                    ).style(container=True)
                                with gr.Column(scale=1):
                                    saveHistoryBtn = gr.Button(i18n("💾 保存对话"))
                                    exportMarkdownBtn = gr.Button(i18n("📝 导出为Markdown"))
                                    gr.Markdown(i18n("默认保存于history文件夹"))
                            with gr.Row(visible=False):
                                with gr.Column():
                                    downloadFile = gr.File(interactive=True)

                with gr.Tab(label=i18n("智人")):
                    systemPromptTxt = gr.Textbox(
                        show_label=True,
                        placeholder=i18n(f"在这里输入System Prompt...\n\n Current system prompt: {INITIAL_SYSTEM_PROMPT}"),
                        label="System prompt",
                        value=INITIAL_SYSTEM_PROMPT,
                        lines=12,
                        visible=False,
                    ).style(container=False)
                    with gr.Accordion(label=i18n("植入角色"), open=True):
                        with gr.Column():
                            with gr.Row():
                                with gr.Column(scale=6):
                                    templateFileSelectDropdown = gr.Dropdown(
                                        label=i18n("选择分类"),
                                        choices=[os.path.splitext(f)[0] for f in get_template_names(plain=True)],
                                        multiselect=False,
                                        value=os.path.splitext(get_template_names(plain=True)[0])[0]
                                    ).style(container=False)
                                with gr.Column(scale=1):
                                    templateRefreshBtn = gr.Button(i18n("🔄 刷新"))
                            with gr.Row():
                                with gr.Column():
                                    templateSelectDropdown = gr.Dropdown(
                                        label=i18n("选择职能"),
                                        choices=load_template(get_template_names(plain=True)[0], mode=1),
                                        multiselect=False,
                                        value=load_template(get_template_names(plain=True)[0], mode=1)[0]
                                    ).style(container=False)
                    model_select_dropdown = gr.Dropdown(
                        label=i18n("选择模型"), choices=MODELS, multiselect=False, value=MODELS[DEFAULT_MODEL],
                        interactive=True, visible=False
                    )
                    lora_select_dropdown = gr.Dropdown(
                        label=i18n("选择LoRA模型"), choices=[], multiselect=False, interactive=True, visible=False
                    )
                    with gr.Row():
                        use_streaming_checkbox = gr.Checkbox(label=i18n("实时传输回答"), value=True, visible=ENABLE_STREAMING_OPTION, interactive=False)
                        single_turn_checkbox = gr.Checkbox(label=i18n("单轮对话"), value=False, visible=False)
                        use_websearch_checkbox = gr.Checkbox(label=i18n("使用在线搜索"), value=False, visible=False)
                        # render_latex_checkbox = gr.Checkbox(label=i18n("渲染LaTeX公式"), value=render_latex, interactive=True, elem_id="render_latex_checkbox")
                    language_select_dropdown = gr.Dropdown(
                        label=i18n("选择回复语言（针对搜索&索引功能）"),
                        choices=REPLY_LANGUAGES,
                        multiselect=False,
                        value=REPLY_LANGUAGES[0],
                        visible=False
                    )
                    index_files = gr.Files(label=i18n("上传"), type="file", visible=False)
                    two_column = gr.Checkbox(label=i18n("双栏pdf"), value=advance_docs["pdf"].get("two_column", False),
                                             visible=False)
                    # TODO: 公式ocr
                    # formula_ocr = gr.Checkbox(label=i18n("识别公式"), value=advance_docs["pdf"].get("formula_ocr", False))

                with gr.Tab(label=i18n("高级")):
                    # gr.Markdown(i18n("# ⚠️ 务必谨慎更改 ⚠️\n\n如果无法使用请恢复默认设置"))
                    # gr.HTML(APPEARANCE_SWITCHER, elem_classes="insert_block")
                    keyTxt = gr.Textbox(
                        show_label=True,
                        placeholder=f"Your API-key...",
                        value=hide_middle_chars(user_api_key.value),
                        type="password",
                        visible=not HIDE_MY_KEY,
                        label="API-Key",
                    )
                    if multi_api_key:
                        usageTxt = gr.Markdown(i18n("多账号模式已开启，无需输入key，可直接开始对话"), elem_id="usage_display",
                                               elem_classes="insert_block")
                    else:
                        usageTxt = gr.Markdown(i18n("**发送消息** 或 **提交key** 以显示额度"), elem_id="usage_display",
                                               elem_classes="insert_block")
                    with gr.Accordion(i18n("参数"), open=True):
                        temperature_slider = gr.Slider(
                            minimum=-0,
                            maximum=2.0,
                            value=1.0,
                            step=0.1,
                            interactive=True,
                            label=i18n("创意度")
                        )
                        top_p_slider = gr.Slider(
                            minimum=-0,
                            maximum=1.0,
                            value=1.0,
                            step=0.05,
                            interactive=True,
                            label=i18n("top-p"),
                            visible=False
                        )
                        n_choices_slider = gr.Slider(
                            minimum=1,
                            maximum=10,
                            value=1,
                            step=1,
                            interactive=True,
                            label=i18n("n choices"),
                            visible=False
                        )
                        stop_sequence_txt = gr.Textbox(
                            show_label=True,
                            placeholder=i18n("在这里输入停止符，用英文逗号隔开..."),
                            label="stop",
                            value="",
                            lines=1,
                            visible=False,
                        )
                        max_context_length_slider = gr.Slider(
                            minimum=1,
                            maximum=32768,
                            value=2000,
                            step=1,
                            interactive=True,
                            label=i18n("max context"),
                            visible=False
                        )
                        max_generation_slider = gr.Slider(
                            minimum=1,
                            maximum=32768,
                            value=1000,
                            step=1,
                            interactive=True,
                            label=i18n("max generations"),
                            visible=False
                        )
                        presence_penalty_slider = gr.Slider(
                            minimum=-2.0,
                            maximum=2.0,
                            value=0.0,
                            step=0.01,
                            interactive=True,
                            label=i18n("presence penalty"),
                            visible=False
                        )
                        frequency_penalty_slider = gr.Slider(
                            minimum=-2.0,
                            maximum=2.0,
                            value=0.0,
                            step=0.01,
                            interactive=True,
                            label=i18n("frequency penalty"),
                            visible=False
                        )
                        logit_bias_txt = gr.Textbox(
                            show_label=True,
                            placeholder=f"word:likelihood",
                            label="logit bias",
                            value="",
                            lines=1,
                            visible=False
                        )
                        user_identifier_txt = gr.Textbox(
                            show_label=True,
                            placeholder=i18n("用于定位滥用行为"),
                            label=i18n("用户名"),
                            value=user_name.value,
                            lines=1,
                            visible=False
                        )

                    with gr.Accordion(i18n("网络设置"), open=False, visible=False):
                        # 优先展示自定义的api_host
                        apihostTxt = gr.Textbox(
                            show_label=True,
                            placeholder=i18n("在这里输入API-Host..."),
                            label="API-Host",
                            value=config.api_host or shared.API_HOST,
                            lines=1
                        )
                        changeAPIURLBtn = gr.Button(i18n("🔄 切换API地址"))
                        proxyTxt = gr.Textbox(
                            show_label=True,
                            placeholder=i18n("在这里输入代理地址..."),
                            label=i18n("代理地址（示例：http://127.0.0.1:10809）"),
                            value="",
                            lines=2
                        )
                        changeProxyBtn = gr.Button(i18n("🔄 设置代理地址"))
                        default_btn = gr.Button(i18n("🔙 恢复默认设置"))

    # Footer
    gr.Markdown(CHUANHU_DESCRIPTION, elem_id="description")
    gr.HTML(FOOTER.format(versions=versions_html()), elem_id="footer")


    def create_greeting(request: gr.Request):
        """
            Access username from gr.Request class. Be able to access current username from application #3259
            https://github.com/gradio-app/gradio/pull/3296
        """
        # Update System Prompt
        show_system_prompt = False
        current_user_prompts = []
        current_user_api_key = ""

        # Update User Profile
        user_request_name = ANONYMOUS_USER  # using anonymous
        if hasattr(request, "username") and request.username:  # is not None or is not ""
            logging.info(f"Get user name from gr.Request: {request.username}")
            if request.username == 'admin001' or request.username == 'admin002' or request.username == 'admin003':
                show_system_prompt = True
            logging.info(f"show_system_prompt: {show_system_prompt}")

            # Update user prompt
            current_user_prompts = load_user_prompts(request.username)

            # Update user API KEY
            current_user_api_key = get_user_key_by_user_name(request.username)
            logging.debug(f"Current user and key pairs: {request.username}:{current_user_api_key}")

            user_request_name = hide_username(request.username)  # just show the last N character
        else:
            logging.info(f"Failed to get user name from gr.Request. ANONYMOUS_USER: {user_request_name}")
            current_user_prompts = load_user_prompts(user_request_name)
        logging.info(f"current_user_prompts: {current_user_prompts}")

        if show_system_prompt:
            user_info_string = gr.Markdown.update(value=i18n(f"Admin: {user_request_name}"))
        else:
            user_info_string = gr.Markdown.update(value=i18n(f"User: {user_request_name}"))

        # Update current model
        first_system_prompt = get_system_prompt_content_by_role_name(current_user_prompts[0])
        current_model = create_new_model(first_system_prompt)
        current_model.set_user_identifier(user_request_name)

        # Update chatBot
        chatbot = gr.Chatbot.update(label=MODELS[DEFAULT_MODEL])

        return user_info_string, user_request_name, \
               current_model, toggle_like_btn_visibility(DEFAULT_MODEL), *current_model.auto_load(), \
               get_history_names(False, user_request_name), chatbot, gr.update(visible=show_system_prompt), \
               gr.update(choices=current_user_prompts, value=current_user_prompts[0]), \
               gr.update(value=current_user_api_key), gr.update(value=current_user_api_key)


    demo.load(fn=create_greeting,
              inputs=None,
              outputs=[user_info, user_name, current_model, like_dislike_area, systemPromptTxt, chatbot,
                       historyFileSelectDropdown, chatbot, systemPromptTxt, templateSelectDropdown, user_api_key, keyTxt],
              api_name="load")

    # Debugging
    '''
    logging.info(
        colorama.Back.GREEN
        + f"\nAfter demo.load() gr.systemPromptTxt: {systemPromptTxt.value}"
        + f"\nAfter demo.load() gr.State.current_prompt_template: {current_prompt_template.value}"
        + f"\nAfter demo.load() gr.State.current_model.system_prompt: {current_model.value.system_prompt}"
        + colorama.Style.RESET_ALL
    )
    '''

    chatgpt_predict_args = dict(
        fn=predict,
        inputs=[
            current_model,
            user_question,
            chatbot,
            use_streaming_checkbox,
            use_websearch_checkbox,
            index_files,
            language_select_dropdown,
        ],
        outputs=[chatbot, status_display],
        show_progress=True,
    )

    start_outputing_args = dict(
        fn=start_outputing,
        inputs=[],
        outputs=[submitBtn, cancelBtn],
        show_progress=True,
    )

    end_outputing_args = dict(
        fn=end_outputing, inputs=[], outputs=[submitBtn, cancelBtn]
    )

    reset_textbox_args = dict(
        fn=reset_textbox, inputs=[], outputs=[user_input]
    )

    transfer_input_args = dict(
        fn=transfer_input, inputs=[user_input], outputs=[user_question, user_input, submitBtn, cancelBtn],
        show_progress=True
    )

    get_usage_args = dict(
        fn=billing_info, inputs=[current_model], outputs=[usageTxt], show_progress=False
    )

    load_history_from_file_args = dict(
        fn=load_chat_history,
        inputs=[current_model, historyFileSelectDropdown, user_name],
        outputs=[saveFileName, systemPromptTxt, chatbot]
    )

    # Chatbot
    cancelBtn.click(interrupt, [current_model], [])
    ''' Running Events Consecutively
    run events consecutively by using the then method of an event listener. This will run an event after the previous 
    event has finished running. This is useful for running events that update components in multiple steps.
    The .then() method of an event listener executes the subsequent event regardless of whether the previous event 
    raised any errors. If you'd like to only run subsequent events if the previous event executed successfully, 
    use the .success() method, which takes the same arguments as .then().
    '''
    user_input.submit(**transfer_input_args).then(**chatgpt_predict_args).then(**end_outputing_args)
    user_input.submit(**get_usage_args)

    submitBtn.click(**transfer_input_args).then(**chatgpt_predict_args, api_name="predict").then(**end_outputing_args)
    submitBtn.click(**get_usage_args)

    index_files.change(handle_file_upload, [current_model, index_files, chatbot],
                       [index_files, chatbot, status_display])

    emptyBtn.click(
        reset,
        inputs=[current_model],
        outputs=[chatbot, status_display],
        show_progress=True,
    )

    retryBtn.click(**start_outputing_args).then(
        retry,
        [
            current_model,
            chatbot,
            use_streaming_checkbox,
            use_websearch_checkbox,
            index_files,
            language_select_dropdown,
        ],
        [chatbot, status_display],
        show_progress=True,
    ).then(**end_outputing_args)
    retryBtn.click(**get_usage_args)

    delFirstBtn.click(
        delete_first_conversation,
        [current_model],
        [status_display],
    )

    delLastBtn.click(
        delete_last_conversation,
        [current_model, chatbot],
        [chatbot, status_display],
        show_progress=False
    )

    likeBtn.click(
        like,
        [current_model],
        [status_display],
        show_progress=False
    )

    dislikeBtn.click(
        dislike,
        [current_model],
        [status_display],
        show_progress=False
    )

    two_column.change(update_doc_config, [two_column], None)

    # LLM Models
    keyTxt.change(set_key, [current_model, keyTxt], [user_api_key, status_display], api_name="set_key").then(
        **get_usage_args)
    keyTxt.submit(**get_usage_args)
    single_turn_checkbox.change(set_single_turn, [current_model, single_turn_checkbox], None)
    model_select_dropdown.change(get_model,
                                 [model_select_dropdown, lora_select_dropdown, user_api_key, temperature_slider,
                                  top_p_slider, systemPromptTxt, user_name],
                                 [current_model, status_display, chatbot, lora_select_dropdown], show_progress=True,
                                 api_name="get_model")
    model_select_dropdown.change(toggle_like_btn_visibility, [model_select_dropdown], [like_dislike_area],
                                 show_progress=False)
    lora_select_dropdown.change(get_model,
                                [model_select_dropdown, lora_select_dropdown, user_api_key, temperature_slider,
                                 top_p_slider, systemPromptTxt, user_name], [current_model, status_display, chatbot],
                                show_progress=True)

    # Template
    systemPromptTxt.change(set_system_prompt, [current_model, systemPromptTxt], None)
    templateRefreshBtn.click(get_template_names_without_extension, None, [templateFileSelectDropdown])
    templateFileSelectDropdown.change(
        load_template,
        [templateFileSelectDropdown],
        [current_prompt_template, templateSelectDropdown],
        show_progress=True,
    )
    templateSelectDropdown.change(
        get_template_content,
        [current_prompt_template, templateSelectDropdown, systemPromptTxt],
        [systemPromptTxt],
        show_progress=True,
    )

    # S&L
    saveHistoryBtn.click(
        save_chat_history,
        [current_model, saveFileName, chatbot, user_name],
        downloadFile,
        show_progress=True,
    )
    saveHistoryBtn.click(get_history_names, [gr.State(False), user_name], [historyFileSelectDropdown])
    exportMarkdownBtn.click(
        export_markdown,
        [current_model, saveFileName, chatbot, user_name],
        downloadFile,
        show_progress=True,
    )
    historyRefreshBtn.click(get_history_names, [gr.State(False), user_name], [historyFileSelectDropdown])
    historyFileSelectDropdown.change(**load_history_from_file_args)
    downloadFile.change(upload_chat_history, [current_model, downloadFile, user_name],
                        [saveFileName, systemPromptTxt, chatbot])

    # Advanced
    max_context_length_slider.change(set_token_upper_limit, [current_model, max_context_length_slider], None)
    temperature_slider.change(set_temperature, [current_model, temperature_slider], None)
    top_p_slider.change(set_top_p, [current_model, top_p_slider], None)
    n_choices_slider.change(set_n_choices, [current_model, n_choices_slider], None)
    stop_sequence_txt.change(set_stop_sequence, [current_model, stop_sequence_txt], None)
    max_generation_slider.change(set_max_tokens, [current_model, max_generation_slider], None)
    presence_penalty_slider.change(set_presence_penalty, [current_model, presence_penalty_slider], None)
    frequency_penalty_slider.change(set_frequency_penalty, [current_model, frequency_penalty_slider], None)
    logit_bias_txt.change(set_logit_bias, [current_model, logit_bias_txt], None)
    user_identifier_txt.change(set_user_identifier, [current_model, user_identifier_txt], None)

    default_btn.click(
        reset_default, [], [apihostTxt, proxyTxt, status_display], show_progress=True
    )
    changeAPIURLBtn.click(
        change_api_host,
        [apihostTxt],
        [status_display],
        show_progress=True,
    )
    changeProxyBtn.click(
        change_proxy,
        [proxyTxt],
        [status_display],
        show_progress=True,
    )

logging.info(
    colorama.Back.GREEN
    + f"\n温馨提示：访问 http://{server_name}:{server_port} 查看界面"
    + colorama.Style.RESET_ALL
)
# 默认开启本地服务器，默认可以直接从IP访问，默认不创建公开分享链接
demo.title = i18n("启源力 AI")

'''Control the rate of processed requests by creating a queue. This will allow you to set the number of requests to 
be processed at one time, and will let users know their position in the queue. Number of worker threads that will be 
processing requests from the queue concurrently. Increasing this number will increase the rate at which requests are 
processed, but will also increase the memory usage of the queue. '''
demo.queue(concurrency_count=CONCURRENT_COUNT)

if __name__ == "__main__":
    reload_javascript()

    # HF deploy
    demo.launch(
        share=False,
        auth=auth_list if authflag else None,
        auth_message=i18n("启源力智人AI助理"),
        favicon_path="./assets/favicon-64.png"
    )

    '''
    # Cloud deploy 
    demo.launch(
        server_name=server_name,
        server_port=server_port,
        share=False,
        auth=auth_list if authflag else None,
        auth_message=i18n("启源力智人AI助理"),
        favicon_path="./assets/favicon-64.png"
    )
    '''

    '''
    # Local deploy
    demo.launch(
        server_name=server_name,
        server_port=server_port,
        share=True,
        auth=auth_list if authflag else None,
        auth_message=i18n("启源力智人AI助理"),
        favicon_path="./assets/favicon-64.png",
        inbrowser=not dockerflag  # 禁止在docker下开启inbrowser
    )
    '''
    # demo.queue(concurrency_count=CONCURRENT_COUNT).launch(
    #     server_name=server_name,
    #     server_port=server_port,
    #     share=share,
    #     auth=auth_list if authflag else None,
    #     favicon_path="./assets/favicon-64.png",
    #     inbrowser=not dockerflag, # 禁止在docker下开启inbrowser
    # )

    # demo.queue(concurrency_count=CONCURRENT_COUNT).launch(server_name="0.0.0.0", server_port=7860, share=False) # 可自定义端口
    # demo.queue(concurrency_count=CONCURRENT_COUNT).launch(server_name="0.0.0.0", server_port=7860,auth=("在这里填写用户名", "在这里填写密码")) # 可设置用户名与密码
    # demo.queue(concurrency_count=CONCURRENT_COUNT).launch(auth=("在这里填写用户名", "在这里填写密码")) # 适合Nginx反向代理
