import gradio as gr
from transformers import Wav2Vec2ForCTC, AutoProcessor
import torch
import librosa
import json

with open('ISO_codes.json', 'r') as file:
    iso_codes = json.load(file)

languages = list(iso_codes.keys())

model_id = "facebook/mms-1b-all"
processor = AutoProcessor.from_pretrained(model_id)
model = Wav2Vec2ForCTC.from_pretrained(model_id)

def transcribe(audio_file_mic=None, audio_file_upload=None, language="Turkmen"):
    if audio_file_mic:
        audio_file = audio_file_mic
    elif audio_file_upload:
        audio_file = audio_file_upload
    else:
        return "Please upload an audio file or record one"

    # Make sure audio is 16kHz
    speech, sample_rate = librosa.load(audio_file)
    if sample_rate != 16000:
        speech = librosa.resample(speech, orig_sr=sample_rate, target_sr=16000)

    # Keep the same model in memory and simply switch out the language adapters by calling load_adapter() for the model and set_target_lang() for the tokenizer
    language_code = ("tuk-script_latin")
    processor.tokenizer.set_target_lang(language_code)
    model.load_adapter(language_code)

    inputs = processor(speech, sampling_rate=16_000, return_tensors="pt")

    with torch.no_grad():
        outputs = model(**inputs).logits

    ids = torch.argmax(outputs, dim=-1)[0]
    transcription = processor.decode(ids)
    return transcription



iface = gr.Interface(fn=transcribe,
                     inputs=[
                         gr.Audio(source="microphone", type="filepath", label="Sesiňi ýazdyr"),
                         gr.Audio(source="upload", type="filepath", label="Sesiňi ýükle"),
                         gr.Dropdown(choices=languages, label="Language", value="Türkmen")
                         ],
                     outputs=gr.Textbox(label="Transcription")
                     )
iface.launch()