import gradio as gr
import edge_tts
import asyncio
import librosa
import soundfile
import io
import argparse
import numpy as np
from inference.infer_tool import Svc

def get_or_create_eventloop():
    try:
        return asyncio.get_event_loop()
    except RuntimeError as ex:
        if "There is no current event loop in thread" in str(ex):
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)
            return asyncio.get_event_loop()

def tts_get_voices_list():
    voices = []
    tts_voice_list = asyncio.get_event_loop().run_until_complete(edge_tts.list_voices())
    for item in tts_voice_list:
        voices.append(item['ShortName'])
    
    return voices

def infer(txt, tts_voice, input_audio, predict_f0, audio_mode):
    if audio_mode:
        if input_audio is  None:
            return 'Please upload your audio file'
        
        sampling_rate, audio = input_audio
        duration = audio.shape[0] / sampling_rate

        if duration > 30:
            return 'The audio file is too long, Please upload audio file that less than 30 seconds'
        
        audio = (audio / np.iinfo(audio.dtype).max).astype(np.float32)
        if len(audio.shape) > 1:
            audio = librosa.to_mono(audio.transpose(1, 0))
        if sampling_rate != 16000:
            audio = librosa.resample(audio, orig_sr=sampling_rate, target_sr=16000)

        raw_path = io.BytesIO()
        soundfile.write(raw_path, audio, 16000, format="wav")
        raw_path.seek(0)
        model = Svc(fr"Herta-Svc/G_10000.pth", f"Herta-Svc/config.json", device = 'cpu')
        out_audio, out_sr = model.infer('speaker0', 0, raw_path, auto_predict_f0 = predict_f0,)
        return (44100, out_audio.cpu().numpy())

    tts = asyncio.run(edge_tts.Communicate(txt, tts_voice).save('audio.mp3'))
    audio, sr = librosa.load('audio.mp3', sr=16000, mono=True)
    raw_path = io.BytesIO()
    soundfile.write(raw_path, audio, 16000, format="wav")
    raw_path.seek(0)
    model = Svc(fr"Herta-Svc/G_10000.pth", f"Herta-Svc/config.json", device = 'cpu')
    out_audio, out_sr = model.infer('speaker0', 0, raw_path, auto_predict_f0 = True,)
    return (44100, out_audio.cpu().numpy())

def change_to_audio_mode(audio_mode):
    if audio_mode:    
        return gr.Audio.update(visible = True), gr.Textbox.update(visible= False), gr.Dropdown.update(visible = False), gr.Checkbox.update(value = True)
    else:
        return gr.Audio.update(visible = False), gr.Textbox.update(visible= True), gr.Dropdown.update(visible = True), gr.Checkbox.update(value = False)

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--device', type=str, default='cpu')
    parser.add_argument('--api', action="store_true", default=False)
    parser.add_argument("--share", action="store_true", default=False, help="share gradio app")
    args = parser.parse_args()
    loop = asyncio.new_event_loop()
    asyncio.set_event_loop(loop)
    with gr.Blocks() as app:
        with gr.Tabs():
            with gr.TabItem('Herta'):
                title = gr.Label('Herta Sovits Model')
                cover = gr.Markdown('<div align="center">'
                            f'<img style="width:auto;height:300px;" src="file/Herta-Svc/herta.png">'
                            '</div>')
                tts_text = gr.Textbox(label="TTS text (100 words limitation)")
                audio_input = gr.Audio(label = 'Please upload audio file that less than 30 seconds', visible = False)
                tts_voice = gr.Dropdown(choices= tts_get_voices_list())
                predict_f0 = gr.Checkbox(label = 'Auto predict F0', value = False)
                audio_mode = gr.Checkbox(label = 'Upload audio instead', value = False)
                audio_output = gr.Audio(label="Output Audio")
                btn_submit = gr.Button("Generate")

        btn_submit.click(infer, [tts_text, tts_voice, audio_input, predict_f0, audio_mode], [audio_output])
        audio_mode.change(change_to_audio_mode, audio_mode, [audio_input, tts_text, tts_voice])
    
    app.queue(concurrency_count=1, api_open=args.api).launch(share=args.share)
