import pandas as pd
import joblib
from fastapi import FastAPI
import uvicorn
import numpy as np
import os

app = FastAPI()

def load_model():
    num_imputer_filepath = "numerical_imputer.joblib"
    scaler_filepath = "scaler.joblib"
    model_filepath = "lr_model.joblib"

    num_imputer = joblib.load(num_imputer_filepath)
    scaler = joblib.load(scaler_filepath)
    model = joblib.load(model_filepath)

    return num_imputer, scaler, model

def preprocess_input_data(input_data, num_imputer, scaler):
    input_data_df = pd.DataFrame([input_data], index=[0])  # Add index [0] to the DataFrame
    input_data_scaled = scaler.transform(input_data_df)
    input_data_scaled = pd.DataFrame(input_data_scaled, columns=input_data_df.columns)
    input_data_imputed = num_imputer.transform(input_data_scaled)

    return input_data_imputed


@app.get("/")
def read_root():
    
    info = """
    Welcome to the Sepsis Prediction API! 🩺💉. This API provides advanced machine learning predictions for sepsis. ⚡📊 For more information and to explore the API's capabilities, please visit the documentation: https://abubakari-sepsis-fastapi-prediction-app.hf.space/docs/
    """
    return info.strip()


@app.get("/sepsis/predict")
def predict_sepsis_endpoint(PRG: float, PL: float, PR: float, SK: float, TS: float,
                            M11: float, BD2: float, Age: float, Insurance: int):
    num_imputer, scaler, model = load_model()

    input_data = {
        'PRG': PRG,
        'PL': PL,
        'PR': PR,
        'SK': SK,
        'TS': TS,
        'M11': M11,
        'BD2': BD2,
        'Age': Age,
        'Insurance': Insurance
    }

    input_scaled_df = preprocess_input_data(input_data, num_imputer, scaler)

    probabilities = model.predict_proba(input_scaled_df)[0]
    prediction = np.argmax(probabilities)

    sepsis_status = "Positive" if prediction == 1 else "Negative"
    
    probability = probabilities[1] if prediction == 1 else probabilities[0]

    #statement = f"The patient is {sepsis_status}. There is a {'high' if prediction == 1 else 'low'} probability ({probability:.2f}) that the patient is susceptible to developing sepsis."

    if prediction == 1:
        status_icon = "✔"  # Red 'X' icon for positive sepsis prediction
        sepsis_explanation = "Sepsis is a life-threatening condition caused by an infection. A positive prediction suggests that the patient might be exhibiting sepsis symptoms and requires immediate medical attention."
    else:
        status_icon = "✘"  # Green checkmark icon for negative sepsis prediction
        sepsis_explanation = "Sepsis is a life-threatening condition caused by an infection. A negative prediction suggests that the patient is not currently exhibiting sepsis symptoms."

    statement = f"The patient's sepsis status is {sepsis_status} {status_icon} with a probability of {probability:.2f}. {sepsis_explanation}"

    user_input_statement = "Please note this is the user-inputted data: "

    output_df = pd.DataFrame([input_data])

    result = {'predicted_sepsis': sepsis_status, 'statement': statement, 'user_input_statement': user_input_statement, 'input_data_df': output_df.to_dict('records')}

    return result

if __name__ == "__main__":
    uvicorn.run(app, host="0.0.0.0", port=7860, reload=True)
