import streamlit as st
from graphviz import Digraph


st.markdown("""
Prompt:
Create an interactive streamlit graph builder using the graphviz diagram model language and the streamlit feature: st.graphviz_chart(figure_or_dot, use_container_width=False) to show an azure cloud architecture model including the top ten architecture components for python full stack development for web, api, ml, models, datasets torch, transformers, streamlit, azure docker and kubernetes pods for scaling

""")

# Dot demo:
import streamlit as st

# Define the default graphviz DOT string
default_dot = """
digraph G {
    rankdir=LR
    node [shape=box]
    WebApp -> API
    API -> Models
    API -> Datasets
    Models -> Torch
    Models -> Transformers
    WebApp -> Streamlit
    Streamlit -> Azure
    Azure -> Docker
    Azure -> Kubernetes
}
"""

# Define the list of top 10 components
components = [
    "WebApp",
    "API",
    "Models",
    "Datasets",
    "Torch",
    "Transformers",
    "Streamlit",
    "Azure",
    "Docker",
    "Kubernetes",
]

# Define a dictionary to map component names to DOT node IDs
node_ids = {
    component: component.lower()
    for component in components
}

def build_dot_string(selected_components):
    """Builds a DOT string representing the selected components"""
    selected_nodes = [node_ids[component] for component in selected_components]
    dot = """
    digraph G {
        rankdir=LR
        node [shape=box]
    """
    for node in selected_nodes:
        dot += f"{node} [color=blue]\n"
    for i in range(len(selected_nodes)):
        for j in range(i+1, len(selected_nodes)):
            dot += f"{selected_nodes[i]} -> {selected_nodes[j]}\n"
    dot += "}"
    return dot

def main():
    st.title("Azure Cloud Architecture Builder")

    # Select the components
    st.sidebar.title("Select components")
    selected_components = st.sidebar.multiselect(
        "Select the top 10 components",
        components,
        default=components[:3]
    )

    # Build the DOT string
    dot = build_dot_string(selected_components)

    # Render the graphviz chart
    st.graphviz_chart(dot, use_container_width=True)

if __name__ == "__main__":
    main()



# Initialize the graph
graph = Digraph(comment='Architectural Model')

# Add nodes to the graph
graph.node('data_layer', 'Data Layer')
graph.node('acr', 'Azure Container Registry')
graph.node('aks', 'Azure Kubernetes\n& Docker Container Pod\nwith Scalability')
graph.node('snowflake', 'Snowflake Instance')
graph.node('cosmos', 'Azure Cosmos\nDatabase')
graph.node('api', 'API Standard\n(using Uvicorn)')
graph.node('soar', 'SOAR Component\n(on Linux Python\nSlimbuster Docker)')

# Add edges to the graph
graph.edge('data_layer', 'acr')
graph.edge('acr', 'aks')
graph.edge('aks', 'snowflake')
graph.edge('aks', 'cosmos')
graph.edge('aks', 'api')
graph.edge('aks', 'soar')

# Define the Streamlit app
def app():
    st.title('Architectural Model')
    
    # Draw the graph
    st.graphviz_chart(graph.source)
    
    # Add buttons to customize the graph
    if st.button('Hide Data Layer'):
        graph.node('data_layer', style='invisible')
    
    if st.button('Hide Snowflake Instance'):
        graph.node('snowflake', style='invisible')
    
    if st.button('Hide SOAR Component'):
        graph.node('soar', style='invisible')



st.markdown("""
# QA Model Spaces:
QA use cases include QA, Semantic Document and FAQ Search.
1. Streamlit Question Answering w Hugging Face: https://huggingface.co/spaces/awacke1/Question-answering
2. Seq2Seq:
	- https://huggingface.co/spaces/awacke1/4-Seq2SeqQAT5
	- https://huggingface.co/spaces/awacke1/AW-04-GR-Seq-2-Seq-QA-Auto-Gen
3. BioGPT: https://huggingface.co/spaces/awacke1/microsoft-BioGPT-Large-PubMedQA
4. NLP QA Context: https://huggingface.co/spaces/awacke1/NLPContextQATransformersRobertaBaseSquad2
	- https://huggingface.co/spaces/awacke1/SOTA-Plan
5.  https://huggingface.co/spaces/awacke1/Question-answering
6.  QA MLM: https://huggingface.co/spaces/awacke1/SOTA-MedEntity
""")


        
# Run the Streamlit app
if __name__ == '__main__':
    app()
