import streamlit as st
import itertools
import nltk
from nltk.corpus import wordnet, words

# Download the necessary resources
nltk.download("wordnet")
nltk.download("words")

def get_related_words(word):
    synonyms = set()
    antonyms = set()
    hypernyms = set()
    hyponyms = set()

    for syn in wordnet.synsets(word):
        for lemma in syn.lemmas():
            synonyms.add(lemma.name())
            if lemma.antonyms():
                antonyms.add(lemma.antonyms()[0].name())

        for hyper in syn.hypernyms():
            hypernyms.update(lemma.name() for lemma in hyper.lemmas())

        for hypo in syn.hyponyms():
            hyponyms.update(lemma.name() for lemma in hypo.lemmas())

    return {
        "synonyms": list(synonyms),
        "antonyms": list(antonyms),
        "hypernyms": list(hypernyms),
        "hyponyms": list(hyponyms),
    }

def generate_words(letters, length=None):
    english_words = set(words.words())
    permutations = set()
    for i in range(1, len(letters) + 1):
        for p in itertools.permutations(letters, i):
            word = "".join(p).lower()  # Convert the word to lowercase
            if (length is None or len(word) == length) and word in english_words:
                permutations.add(word)
    return permutations


st.title("Scrabble Helper")

letters = st.text_input("Enter the letters you have:")
word_length = st.number_input("Enter the word length (optional):", min_value=0, value=0, step=1)

if letters:
    st.header("Generated Words")
    words = generate_words(letters, length=word_length if word_length > 0 else None)
    st.write(words)

    st.header("Thesaurus and Related Words Lookup")
    selected_word = st.selectbox("Select a word to look up related words:", [""] + sorted(words))
    if selected_word:
        related_words = get_related_words(selected_word)
        st.subheader("Synonyms")
        st.write(related_words["synonyms"])
        st.subheader("Antonyms")
        st.write(related_words["antonyms"])
        st.subheader("Hypernyms (more general terms)")
        st.write(related_words["hypernyms"])
        st.subheader("Hyponyms (more specific terms)")
        st.write(related_words["hyponyms"])
