from langchain.prompts import PromptTemplate
from langchain.chains import LLMChain
from langchain.agents import tool
from agents.tools.smart_domain.common import getPrefix
from models import llm

entity_architecture = """
Entity: This component is use to represents business concepts and encapsulates business rules.
It may include 3 parts: 
- id(identity of entity)
- description (properties package of entity represent the value of entity), 
- associations (collection of associated entiy)
---example code:
    @Getter
    @AllArgsConstructor 
    public class Feature {{
        // id
        private FeatureId id;
        
        // description
        private FeatureDescription description;

        // associations
        private FeatureConfigs configs;

        public record FeatureId(String featureKey) {{
        
        }}

        @Builder
        public record FeatureDescription(String name, 
                            String description, 
                            Boolean isEnable,
                            LocalDateTime updatedAt,
                            LocalDateTime createdAt))) {{
        
        }}

        public Feature update(Feature newFeature) {{
            this.description = FeatureDescription.builder()
                .name(newFeature.description.name())
                .description(newFeature.description.description())
                .isEnable(this.description.isEnable())
                .updatedAt(LocalDateTime.now())
                .createdAt(this.description.createdAt());

            return this;
        }}

        public interface FeatureConfigs() {{
            Flux<FeatureConfig> findAll();
            Flux<FeatureConfig> subCollection(long from, long to);
            Mono<FeatureConfig> findById(FeatureConfigId id);
        }}
    }}
---end of example code
"""

entity_test_strategy = """
For the Entity, we can write unit test to ensure that the business rules related to Merchandise are correctly encapsulated.
---example code
    class FeatureTest {{
        @Test
        void should_update_feature_description() {{
            // given
            Feature feature = Feature.builder()
                .id(new FeatureId("featureKey"))
                .description(new FeatureDescription("name", "description", true, LocalDateTime.now(), LocalDateTime.now()))
                .build();
            Feature newFeature = Feature.builder()
                .id(new FeatureId("featureKey"))
                .description(new FeatureDescription("newName", "newDescription", true, LocalDateTime.now(), LocalDateTime.now()))
                .build();
            // when
            feature.update(newFeature);
            // then
            assertThat(feature.description().name()).isEqualTo("newName");
            assertThat(feature.description().description()).isEqualTo("newDescription");
        }}
    }}
---end of example code
"""

entity_tech_stack = """
Java17、reactor、lombok、Junit5、reactor test、Mockito
"""

entity_task = """Your task is to generate the Enity of domain layer tests and product code."""
ENTITY = getPrefix(entity_task, entity_tech_stack, entity_architecture, entity_test_strategy) + """

Use the following format:
request: the request that you need to fulfill

Entity:
```
the Entity code that you write to fulfill the request, follow TechStack and Architecture
```

Test:
```
the test code that you write to fulfill the request, follow TechStack Architecture and TestStrategy
```

request: {input}"""

ENTITY_PROMPT = PromptTemplate(input_variables=["input"], template=ENTITY,)

entityChain = LLMChain(llm = llm(temperature=0.1), prompt=ENTITY_PROMPT)


@tool("Generate Entity Code", return_direct=True)
def entityCodeGenerator(input: str) -> str:
    '''useful for when you need to generate entity code'''
    response = entityChain.run(input)
    return response