import pickle, joblib
import gradio as gr
from datetime import datetime, timedelta, timezone

model = joblib.load('model.pkl')

def preprocess_city(selected_city):
    # Map the selected city to its one-hot encoded representation
    city_mapping = {
        'Hyderabad' : [1, 0, 0, 0, 0, 0, 0],
        'Indore': [1, 0, 0, 0, 0, 0, 0],
        'Jaipur': [0, 1, 0, 0, 0, 0, 0],
        'Mahabaleshwar': [0, 0, 1, 0, 0, 0, 0],
        'Mussoorie': [0, 0, 0, 1, 0, 0, 0],
        'Raipur': [0, 0, 0, 0, 1, 0, 0],
        'Udaipur': [0, 0, 0, 0, 0, 1, 0],
        'Varanasi': [0, 0, 0, 0, 0, 0, 1]
    }
    return city_mapping[selected_city]

def preprocess_date(date_string):
    # Parse the date string into a datetime object
    date_obj = datetime.strptime(date_string, '%Y-%m-%d')
    year = date_obj.year
    month = date_obj.month
    day = date_obj.day
    return year, month, day

def calculate_lead_time(checkin_date):
    # Convert input date to datetime object
    input_date = datetime.strptime(checkin_date, '%Y-%m-%d')
    
    # Get current date and time in GMT+5:30 timezone
    current_date = datetime.now(timezone(timedelta(hours=5, minutes=30)))
    
    # Make current_date an aware datetime with the same timezone
    current_date = current_date.replace(tzinfo=input_date.tzinfo)
    
    # Calculate lead time as difference in days
    lead_time = (input_date - current_date).days
    
    return lead_time

def is_weekend(checkin_date):
    # Convert input date to datetime object
    input_date = datetime.strptime(checkin_date, '%Y-%m-%d')
    
    # Calculate the day of the week (0=Monday, 6=Sunday)
    day_of_week = input_date.weekday()
    
    # Check if the day is Friday (4) or Saturday (5)
    return 1 if day_of_week == 4 or day_of_week == 5 else 0

def predict(selected_city, checkin_date, star_rating, text_rating, season, additional_views, room_category):
    # Preprocess user input
    # Here, selected_city is the name of the city selected from the dropdown
    # checkin_date is the date selected using the text input
    # star_rating is the selected star rating from the dropdown
    # text_rating is the numeric rating from the text box
    # season is the selected option from the radio button (On Season or Off Season)
    season_binary = 1 if season == 'On Season' else 0
    # additional_views is the selected option from the radio button (Yes or No)
    additional_views_binary = 1 if additional_views == 'Yes' else 0
    
    room_categories = ["Dorm", "Standard", "Deluxe", "Executive", "Suite"]
    room_category_number = room_categories.index(room_category)
    
    # Preprocess the date
    year, month, day = preprocess_date(checkin_date)
    
    # Preprocess the selected city
    city_encoded = preprocess_city(selected_city)
    
    # Calculate lead time
    lead_time = calculate_lead_time(checkin_date)
    
    # Calculate if the input date is a weekend (1) or weekday (0)
    is_weekend_value = is_weekend(checkin_date)
    
    # Combine all the input features
    input_data = [star_rating, text_rating, season_binary, day, month, year, is_weekend_value, lead_time,room_category_number, additional_views_binary]+city_encoded

    # Make predictions using the model
    prediction = model.predict([input_data])
    return "{:.2f}".format(prediction[0])

# Define input components
city_dropdown = gr.components.Dropdown(choices=['Hyderabad', 'Indore', 'Jaipur', 'Mahabaleshwar', 'Mussoorie', 'Raipur', 'Udaipur', 'Varanasi'], label='Select a City')
date_input = gr.components.Textbox(label='Check-in Date (YYYY-MM-DD)')
star_rating_dropdown = gr.components.Dropdown(choices=[1, 2, 3, 4, 5], label='Select Star Rating')
text_rating_input = gr.components.Number(label='Enter Numeric Rating (1-5)')
season_radio = gr.components.Radio(['On Season', 'Off Season'], label='Season')
room_category_dropdown = gr.components.Dropdown(choices=["Dorm", "Standard", "Deluxe", "Executive", "Suite"], label='Select Room Category')
additional_views_radio = gr.components.Radio(['Yes', 'No'], label='Additional Views')

# Define output component
output = gr.components.Textbox(label='Predicted Output')
# Create the interface
interface = gr.Interface(fn=predict, inputs=[city_dropdown, date_input, star_rating_dropdown, text_rating_input, season_radio, additional_views_radio, room_category_dropdown], outputs=output, title='Model Prediction Interface')

# Launch the interface
interface.launch()

