import streamlit as st
import requests
from wordpress_xmlrpc import Client, WordPressPost
from wordpress_xmlrpc.methods.posts import NewPost
import os
from dotenv import load_dotenv

load_dotenv()
openai_api_key = os.getenv("OPENAI_API_KEY")
wp_url = f"{os.getenv('WP_URL')}/xmlrpc.php"
wp_username = os.getenv("WP_USERNAME")
wp_password = os.getenv("WP_PASSWORD")

if openai_api_key:

    def get_filetext(filename, cache={}):
        if filename not in cache:
            if not os.path.exists(filename):
                raise ValueError(f"ファイル '{filename}' が見つかりませんでした")
            with open(filename, "r") as f:
                cache[filename] = f.read()
        return cache[filename]

    def generate_blog_post(prompt):
        constraints = get_filetext(filename="constraints.md")
        
        data = {
            "model": "gpt-4",
            "messages": [
                {"role": "system", "content": constraints},
                {"role": "user", "content": prompt},
            ],
            "max_tokens": 1024,
            "n": 1,
            "stop": None,
            "temperature": 0.7,
        }

        response = requests.post(
            "https://api.openai.com/v1/chat/completions",
            headers={
                "Content-Type": "application/json",
                "Authorization": f"Bearer {openai_api_key}"
            },
            json=data
        )

        response.raise_for_status()
        choice = response.json()['choices'][0]
        blog_text = choice['message']['content'].strip()
        return blog_text

    def post_to_wordpress(title, content):
        client = Client(wp_url, wp_username, wp_password)
        post = WordPressPost()
        post.title = title
        post.content = content
        post.post_status = "publish"
        post_id = client.call(NewPost(post))
        return post_id

    st.title("ChatGPTによるブログ記事生成")
    prompt = st.text_input("記事のタイトルを入力してください:")

    generated_post = st.session_state.get("generated_post", None)

    if st.button("記事生成"):
        generated_post = generate_blog_post(prompt)
        st.session_state.generated_post = generated_post
        st.write("生成されたブログ記事:")
        st.write(generated_post)

    if generated_post:
        if st.button("投稿"):
            post_id = post_to_wordpress(prompt, generated_post)
            st.write(f"ブログ記事が投稿されました。記事ID: {post_id}")

else:
    st.write("サービスを利用するためには、このスペースを複製し、以下の環境変数を定義してください。設定方法はosenv_setting_tips.txtを参照してください。")
    st.write("OPENAI_API_KEY, WP_URL, WP_USERNAME, WP_PASSWORD")

st.markdown(
    """
    <h3>注意事項</h3>
    <ol>
        <li style="font-size: small;">投稿前に記事の内容をよく確認してください。</li>
        <li style="font-size: small;">OpenAIのAPIキーや、WordPressのURL、ユーザーID、パスワードはシステム設定にて設定します。詳しくはosenv_setting_tips.txtを参照ください。</li>
        <li style="font-size: small;">constraints.mdを修正すると、生成される記事の内容、雰囲気をカスタマイズすることが可能です。</li>
        <li style="font-size: small;">当サービスでは、OpenAI社のChatGPT APIのgpt-4を使用しております。</li>
        <li style="font-size: small;">当サービスで生成されたコンテンツは、OpenAI が提供する人工知能によるものであり、当サービスやOpenAI がその正確性や信頼性を保証するものではありません。</li>
        <li style="font-size: small;"><a href="https://platform.openai.com/docs/usage-policies">OpenAI の利用規約</a>に従い、データ保持しない方針です（ただし諸般の事情によっては変更する可能性はございます）。
        <li style="font-size: small;">当サービスで生成されたコンテンツは事実確認をした上で、コンテンツ生成者およびコンテンツ利用者の責任において利用してください。</li>
        <li style="font-size: small;">当サービスでの使用により発生したいかなる損害についても、当社は一切の責任を負いません。</li>
        <li style="font-size: small;">当サービスはβ版のため、予告なくサービスを終了する場合がございます。</li>
    </ol>
    <h3>謝辞</h3>
    <ol>
        <li style="font-size: small;">このサービスは<a href="https://huggingface.co/spaces/shigel/aiemo" target="_blank">aiemo</a>を参考に作成しました。大変感謝しております！特に、性格設定のアイデアは秀逸です。ありがとうございました！</li>
    </ol>
    """,
    unsafe_allow_html=True,
)

st.markdown(
    f'<a href="https://huggingface.co/spaces/7thHeaven/GPT2WordPress?duplicate=true">'
    f'<img src="https://bit.ly/3gLdBN6" alt="Duplicate Space"></a>',
    unsafe_allow_html=True,
)
        
