#!/usr/bin/env python

import os
import random

import gradio as gr
import numpy as np
import PIL.Image
import torch
import torchvision.transforms.functional as TF
from diffusers import (
    AutoencoderKL,
    EulerAncestralDiscreteScheduler,
    StableDiffusionXLAdapterPipeline,
    T2IAdapter,
)

from modelscope.pipelines import pipeline
from modelscope.outputs import OutputKeys

DESCRIPTION = '''# doodle2vid
Combining T2I-Adapter-SDXL with MS-Image2Video to create a doodle to video pipeline.
Shout-out to [fffiloni](https://huggingface.co/fffiloni) & [ARC Lab, Tencent PCG](https://huggingface.co/TencentARC) 🗣️

How to use: Draw a doodle in the canvas, and click "Run" to generate a video.
You can also provide a prompt with more details and choose a style.
'''

if not torch.cuda.is_available():
    DESCRIPTION += "\n<p>Running on CPU 🥶 This demo does not work on CPU.</p>"

style_list = [
    {
        "name": "(No style)",
        "prompt": "{prompt}",
        "negative_prompt": "",
    },
    {
        "name": "Cinematic",
        "prompt": "cinematic still {prompt} . emotional, harmonious, vignette, highly detailed, high budget, bokeh, cinemascope, moody, epic, gorgeous, film grain, grainy",
        "negative_prompt": "anime, cartoon, graphic, text, painting, crayon, graphite, abstract, glitch, deformed, mutated, ugly, disfigured",
    },
    {
        "name": "3D Model",
        "prompt": "professional 3d model {prompt} . octane render, highly detailed, volumetric, dramatic lighting",
        "negative_prompt": "ugly, deformed, noisy, low poly, blurry, painting",
    },
    {
        "name": "Anime",
        "prompt": "anime artwork {prompt} . anime style, key visual, vibrant, studio anime,  highly detailed",
        "negative_prompt": "photo, deformed, black and white, realism, disfigured, low contrast",
    },
    {
        "name": "Digital Art",
        "prompt": "concept art {prompt} . digital artwork, illustrative, painterly, matte painting, highly detailed",
        "negative_prompt": "photo, photorealistic, realism, ugly",
    },
    {
        "name": "Photographic",
        "prompt": "cinematic photo {prompt} . 35mm photograph, film, bokeh, professional, 4k, highly detailed",
        "negative_prompt": "drawing, painting, crayon, sketch, graphite, impressionist, noisy, blurry, soft, deformed, ugly",
    },
    {
        "name": "Pixel art",
        "prompt": "pixel-art {prompt} . low-res, blocky, pixel art style, 8-bit graphics",
        "negative_prompt": "sloppy, messy, blurry, noisy, highly detailed, ultra textured, photo, realistic",
    },
    {
        "name": "Fantasy art",
        "prompt": "ethereal fantasy concept art of  {prompt} . magnificent, celestial, ethereal, painterly, epic, majestic, magical, fantasy art, cover art, dreamy",
        "negative_prompt": "photographic, realistic, realism, 35mm film, dslr, cropped, frame, text, deformed, glitch, noise, noisy, off-center, deformed, cross-eyed, closed eyes, bad anatomy, ugly, disfigured, sloppy, duplicate, mutated, black and white",
    },
    {
        "name": "Neonpunk",
        "prompt": "neonpunk style {prompt} . cyberpunk, vaporwave, neon, vibes, vibrant, stunningly beautiful, crisp, detailed, sleek, ultramodern, magenta highlights, dark purple shadows, high contrast, cinematic, ultra detailed, intricate, professional",
        "negative_prompt": "painting, drawing, illustration, glitch, deformed, mutated, cross-eyed, ugly, disfigured",
    },
    {
        "name": "Manga",
        "prompt": "manga style {prompt} . vibrant, high-energy, detailed, iconic, Japanese comic style",
        "negative_prompt": "ugly, deformed, noisy, blurry, low contrast, realism, photorealistic, Western comic style",
    },
]

styles = {k["name"]: (k["prompt"], k["negative_prompt"]) for k in style_list}
STYLE_NAMES = list(styles.keys())
DEFAULT_STYLE_NAME = "(No style)"


def apply_style(style_name: str, positive: str, negative: str = "") -> tuple[str, str]:
    p, n = styles.get(style_name, styles[DEFAULT_STYLE_NAME])
    return p.replace("{prompt}", positive), n + negative


device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
if torch.cuda.is_available():
    model_id = "stabilityai/stable-diffusion-xl-base-1.0"
    adapter = T2IAdapter.from_pretrained(
        "TencentARC/t2i-adapter-sketch-sdxl-1.0", torch_dtype=torch.float16, variant="fp16"
    )
    scheduler = EulerAncestralDiscreteScheduler.from_pretrained(model_id, subfolder="scheduler")
    pipe = StableDiffusionXLAdapterPipeline.from_pretrained(
        model_id,
        vae=AutoencoderKL.from_pretrained("madebyollin/sdxl-vae-fp16-fix", torch_dtype=torch.float16),
        adapter=adapter,
        scheduler=scheduler,
        torch_dtype=torch.float16,
        variant="fp16",
    )
    pipe.to(device)
else:
    pipe = None

MAX_SEED = np.iinfo(np.int32).max
video_pipe = pipeline(task='image-to-video', model='damo/Image-to-Video', model_revision='v1.1.0')


def randomize_seed_fn(seed: int, randomize_seed: bool) -> int:
    if randomize_seed:
        seed = random.randint(0, MAX_SEED)
    return seed

def inferVideo(image: PIL.Image.Image) -> str:
    # Save the passed image to a temp file
    temp_path = "temp_input_image.png"
    image.save(temp_path)
    
    output_video_path = video_pipe(temp_path, output_video='output.mp4')[OutputKeys.OUTPUT_VIDEO]
    print(output_video_path)
    return output_video_path

def inferImage(
    image: PIL.Image.Image,
    prompt: str,
    negative_prompt: str,
    style_name: str = DEFAULT_STYLE_NAME,
    num_steps: int = 25,
    guidance_scale: float = 5,
    adapter_conditioning_scale: float = 0.8,
    adapter_conditioning_factor: float = 0.8,
    seed: int = 0,
    progress=gr.Progress(track_tqdm=True),
) -> PIL.Image.Image:
    image = image.convert("RGB")
    image = TF.to_tensor(image) > 0.5
    image = TF.to_pil_image(image.to(torch.float32))

    prompt, negative_prompt = apply_style(style_name, prompt, negative_prompt)

    generator = torch.Generator(device=device).manual_seed(seed)
    out = pipe(
        prompt=prompt,
        negative_prompt=negative_prompt,
        image=image,
        num_inference_steps=num_steps,
        generator=generator,
        guidance_scale=guidance_scale,
        adapter_conditioning_scale=adapter_conditioning_scale,
        adapter_conditioning_factor=adapter_conditioning_factor,
    ).images[0]
    
    return out


with gr.Blocks(css="style.css") as demo:
    gr.Markdown(DESCRIPTION, elem_id="description")

    with gr.Row():
        with gr.Column():
            with gr.Group():
                image = gr.Image(
                    source="canvas",
                    tool="sketch",
                    type="pil",
                    image_mode="L",
                    invert_colors=True,
                    shape=(1024, 1024),
                    brush_radius=4,
                    height=440,
                )
                prompt = gr.Textbox(label="Prompt")
                style = gr.Dropdown(label="Style", choices=STYLE_NAMES, value=DEFAULT_STYLE_NAME)
                run_button = gr.Button("Run")
            with gr.Accordion("Advanced options", open=False):
                negative_prompt = gr.Textbox(
                    label="Negative prompt",
                    value=" extra digit, fewer digits, cropped, worst quality, low quality, glitch, deformed, mutated, ugly, disfigured",
                )
                num_steps = gr.Slider(
                    label="Number of steps",
                    minimum=1,
                    maximum=50,
                    step=1,
                    value=25,
                )
                guidance_scale = gr.Slider(
                    label="Guidance scale",
                    minimum=0.1,
                    maximum=10.0,
                    step=0.1,
                    value=5,
                )
                adapter_conditioning_scale = gr.Slider(
                    label="Adapter conditioning scale",
                    minimum=0.5,
                    maximum=1,
                    step=0.1,
                    value=0.8,
                )
                adapter_conditioning_factor = gr.Slider(
                    label="Adapter conditioning factor",
                    info="Fraction of timesteps for which adapter should be applied",
                    minimum=0.5,
                    maximum=1,
                    step=0.1,
                    value=0.8,
                )
                seed = gr.Slider(
                    label="Seed",
                    minimum=0,
                    maximum=MAX_SEED,
                    step=1,
                    value=0,
                )
                randomize_seed = gr.Checkbox(label="Randomize seed", value=True)
        with gr.Column():
            result_image = gr.Image(label="Intermediate Image Output", type="pil", interactive=False, height=400)
            result_video = gr.Video(label="Final Video Output", height=400)

    inputs = [
        image,
        prompt,
        negative_prompt,
        style,
        num_steps,
        guidance_scale,
        adapter_conditioning_scale,
        adapter_conditioning_factor,
        seed,
    ]
    prompt.submit(
        fn=randomize_seed_fn,
        inputs=[seed, randomize_seed],
        outputs=seed,
        queue=False,
        api_name=False,
    ).then(
        fn=inferImage,
        inputs=inputs,
        outputs=result_image,
        api_name=False,
    ).then(
        fn=inferVideo,
        inputs=[result_image],
        outputs=result_video,
        api_name=False,
    )
    negative_prompt.submit(
        fn=randomize_seed_fn,
        inputs=[seed, randomize_seed],
        outputs=seed,
        queue=False,
        api_name=False,
    ).then(
        fn=inferImage,
        inputs=inputs,
        outputs=result_image,
        api_name=False,
    ).then(
        fn=inferVideo,
        inputs=[result_image],
        outputs=result_video,
        api_name=False,
    )
    run_button.click(
        fn=randomize_seed_fn,
        inputs=[seed, randomize_seed],
        outputs=seed,
        queue=False,
        api_name=False,
    ).then(
        fn=inferImage,
        inputs=inputs,
        outputs=result_image,
        api_name=False,
    ).then(
        fn=inferVideo,
        inputs=[result_image],
        outputs=result_video,
        api_name=False,
    )

if __name__ == "__main__":
    demo.queue(max_size=20).launch()
