import os
import tempfile
import uuid
import subprocess
import gradio as gr


BIN = os.path.join(os.path.dirname(__file__), "bin", "zkProver_linux_gpu")


def run_zk_prover(network, block_number, contract, file):
    if not contract:
        raise gr.Error("contract is required")
    if not file:
        raise gr.Error('file is required')
    args = [
        BIN,
        "evm", "-r", "https://rpc.flashbots.net/"
    ]
    if block_number:
        args.extend(["-b", str(block_number)])
    proof_path = "/tmp/" + str(uuid.uuid4()) + ".bin"
    args.extend(["-o", proof_path])

    args.append(file.name + ":" + contract)

    proc = subprocess.Popen(args,)
    proc.wait()

    if proc.returncode != 0:
        raise gr.Error("generate proof failed")
    return proof_path


with gr.Blocks() as demo:
    gr.Markdown(
    """
        # 0xHacked
        This is the demo for [0xHacked](https://0xHacked.com), a trustless bug bounty platform. You can generate the proof of exploit here. However, due to the constraints of ZKP, the generation might be low on Huggingface.
        <br/>
        We recommend [compiling it from the source](https://github.com/0xHackedLabs/zkProver). The generation can be very quick on GPU. For more details, please refer to [0xHacked Documentation](https://docs.0xHacked.com).
        <br/>
        The sample PoC provided below takes ~800s to generate the proof. You can click "SushiRouterExploit.sol" below and hit "Run" to try it!
        """
    )
    with gr.Column():
        with gr.Row():
            with gr.Column():
                network_input = gr.Dropdown(["Ethereum"], value="Ethereum",  label='Network')
                block_number_input = gr.Number(precision=0, label='Block Number')
                contract_input = gr.Textbox(label='Poc Contract')
                file_input = gr.File(file_types=[".sol"], label='Solidity File')
                submit_btn = gr.Button(label="Submit")
            with gr.Column():
                fileout = gr.File(label='Proof File')

    gr.Examples(
        examples=[[
            "Ethereum",
            17007841,
            "SushiExpProxy",
            "./examples/SushiRouterExploit.sol"],
        ],
        fn=run_zk_prover,
        inputs=[network_input, block_number_input, contract_input, file_input],
        outputs=fileout
    )

    submit_btn.click(
        fn=run_zk_prover,
        inputs=[network_input, block_number_input, contract_input, file_input],
        outputs=fileout
    )

if __name__ == "__main__":
    demo.launch(server_name="0.0.0.0", server_port=7860)

