/*
 * DBeaver - Universal Database Manager
 * Copyright (C) 2010-2023 DBeaver Corp and others
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jkiss.dbeaver.ui.views.session;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.*;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchSite;
import org.eclipse.ui.PartInitException;
import org.jkiss.code.NotNull;
import org.jkiss.code.Nullable;
import org.jkiss.dbeaver.Log;
import org.jkiss.dbeaver.model.DBIcon;
import org.jkiss.dbeaver.model.DBPObject;
import org.jkiss.dbeaver.model.DBPObjectWithDescription;
import org.jkiss.dbeaver.model.DBUtils;
import org.jkiss.dbeaver.model.admin.sessions.*;
import org.jkiss.dbeaver.model.exec.DBCExecutionContext;
import org.jkiss.dbeaver.model.exec.DBCExecutionPurpose;
import org.jkiss.dbeaver.model.exec.DBCSession;
import org.jkiss.dbeaver.model.exec.plan.DBCQueryPlanner;
import org.jkiss.dbeaver.model.preferences.DBPPropertyDescriptor;
import org.jkiss.dbeaver.model.runtime.DBRProgressMonitor;
import org.jkiss.dbeaver.model.runtime.load.DatabaseLoadService;
import org.jkiss.dbeaver.model.sql.SQLQuery;
import org.jkiss.dbeaver.model.sql.SQLUtils;
import org.jkiss.dbeaver.runtime.DBWorkbench;
import org.jkiss.dbeaver.runtime.properties.ObjectPropertyDescriptor;
import org.jkiss.dbeaver.runtime.properties.PropertyCollector;
import org.jkiss.dbeaver.ui.*;
import org.jkiss.dbeaver.ui.controls.ListContentProvider;
import org.jkiss.dbeaver.ui.controls.ProgressLoaderVisualizer;
import org.jkiss.dbeaver.ui.controls.autorefresh.AutoRefreshControl;
import org.jkiss.dbeaver.ui.editors.StringEditorInput;
import org.jkiss.dbeaver.ui.editors.SubEditorSite;
import org.jkiss.dbeaver.ui.editors.sql.SQLEditorBase;
import org.jkiss.dbeaver.ui.editors.sql.handlers.SQLEditorHandlerOpenObjectConsole;
import org.jkiss.dbeaver.ui.editors.sql.handlers.SQLNavigatorContext;
import org.jkiss.dbeaver.ui.editors.sql.plan.ExplainPlanViewer;
import org.jkiss.dbeaver.ui.navigator.itemlist.DatabaseObjectListControl;
import org.jkiss.dbeaver.ui.properties.PropertyTreeViewer;
import org.jkiss.dbeaver.ui.views.session.internal.SessionEditorMessages;
import org.jkiss.dbeaver.utils.GeneralUtils;
import org.jkiss.utils.CommonUtils;

import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * SessionManagerViewer
 */
public class SessionManagerViewer<SESSION_TYPE extends DBAServerSession>
{
    private static final Log log = Log.getLog(SessionManagerViewer.class);

    private final IWorkbenchPart workbenchPart;
    private final DBAServerSessionManager<SESSION_TYPE> sessionManager;
    private final SessionListControl sessionTable;
    //private Text sessionInfo;
    private final IEditorSite subSite;
    private final SQLEditorBase sqlViewer;

    private final Font boldFont;
    private final PropertyTreeViewer sessionProps;
    private DBAServerSession curSession;
    private final AutoRefreshControl refreshControl;
    private final SashForm sashMain;
    private final SashForm sashDetails;

    private IDialogSettings settings;

    private final CTabFolder previewFolder;
    private final CTabItem detailsItem;

    private final DBCQueryPlanner planner;
    private ExplainPlanViewer planViewer;
    private Object selectedPlanElement;
    private final CTabFolder detailsFolder;

    protected SessionManagerViewer(IWorkbenchPart part, Composite parent, final DBAServerSessionManager<SESSION_TYPE> sessionManager) {
        this.workbenchPart = part;
        this.sessionManager = sessionManager;
        this.subSite = new SubEditorSite(workbenchPart.getSite());
        this.boldFont = UIUtils.makeBoldFont(parent.getFont());

        planner = DBUtils.getAdapter(DBCQueryPlanner.class, sessionManager.getDataSource());

        Composite composite = UIUtils.createPlaceholder(parent, 1);

        sashMain = UIUtils.createPartDivider(workbenchPart, composite, SWT.VERTICAL | SWT.SMOOTH);
        sashMain.setLayoutData(new GridData(GridData.FILL_BOTH));

        refreshControl = new AutoRefreshControl(sashMain, sessionManager.getClass().getSimpleName(), monitor -> UIUtils.syncExec(this::refreshSessions));

        {
            sessionTable = new SessionListControl(sashMain, workbenchPart.getSite(), sessionManager);
            sessionTable.getItemsViewer().addSelectionChangedListener(event -> onSessionSelect(getSelectedSession()));
            sessionTable.addDisposeListener(e -> saveSettings(settings));

            sessionTable.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
            sessionTable.createProgressPanel(composite);
        }

        {
            sashDetails = UIUtils.createPartDivider(workbenchPart, sashMain, SWT.HORIZONTAL | SWT.SMOOTH);
            sashDetails.setLayoutData(new GridData(GridData.FILL_BOTH));

            {
                previewFolder = new CTabFolder(sashDetails, SWT.TOP);
                sqlViewer = new SQLEditorBase() {
                    @Override
                    public DBCExecutionContext getExecutionContext() {
                        return DBUtils.getDefaultContext(sessionManager.getDataSource(), false);
                    }

                    @Override
                    public boolean isFoldingEnabled() {
                        return false;
                    }
                };
                updateSQL();
                sqlViewer.createPartControl(previewFolder);
                Object text = sqlViewer.getAdapter(Control.class);
                if (text instanceof StyledText) {
                    ((StyledText) text).setWordWrap(true);
                }

                sqlViewer.reloadSyntaxRules();

                parent.addDisposeListener(e -> sqlViewer.dispose());

                CTabItem sqlViewItem = new CTabItem(previewFolder, SWT.NONE);
                sqlViewItem.setText(SessionEditorMessages.viewer_view_item_sql);
                sqlViewItem.setImage(DBeaverIcons.getImage(UIIcon.SQL_TEXT));
                sqlViewItem.setControl(sqlViewer.getEditorControlWrapper());

                previewFolder.setSelection(sqlViewItem);

                if (planner != null) {
                    createPlannerTab(previewFolder);
                }

                previewFolder.addSelectionListener(new SelectionAdapter() {
                    @Override
                    public void widgetSelected(SelectionEvent e) {
                        updatePreview();
                    }
                });
            }

            {
                detailsFolder = new CTabFolder(sashDetails, SWT.TOP);
                sessionProps = new PropertyTreeViewer(detailsFolder, SWT.NONE);

                detailsItem = new CTabItem(detailsFolder, SWT.NONE);
                detailsItem.setText(SessionEditorMessages.viewer_details_item_details);
                detailsItem.setImage(DBeaverIcons.getImage(UIIcon.PROPERTIES));
                detailsItem.setControl(sessionProps.getControl());

                if (sessionManager instanceof DBAServerSessionDetailsProvider) {
                    List<DBAServerSessionDetails> sessionDetails = ((DBAServerSessionDetailsProvider) sessionManager).getSessionDetails();
                    if (sessionDetails != null) {
                        for (DBAServerSessionDetails detailsInfo : sessionDetails) {
                            CTabItem extDetailsItem = new CTabItem(detailsFolder, SWT.NONE);
                            extDetailsItem.setData(detailsInfo);
                            extDetailsItem.setText(detailsInfo.getDetailsTitle());
                            if (detailsInfo.getDetailsIcon() != null) {
                                extDetailsItem.setImage(DBeaverIcons.getImage(detailsInfo.getDetailsIcon()));
                            }
                            if (detailsInfo.getDetailsTooltip() != null) {
                                extDetailsItem.setToolTipText(detailsInfo.getDetailsTooltip());
                            }

                            Class<?> detailsType = detailsInfo.getDetailsType();
                            if (DBPObjectWithDescription.class.isAssignableFrom(detailsType)) {
                                StyledText text = new StyledText(detailsFolder, SWT.READ_ONLY | SWT.V_SCROLL | SWT.H_SCROLL);
                                text.setForeground(UIStyles.getDefaultTextForeground());
                                text.setBackground(UIStyles.getDefaultTextBackground());
                                text.setFont(UIUtils.getMonospaceFont());
                                text.setData(detailsInfo);
                                extDetailsItem.setControl(text);
                            } else if (DBPObject.class.isAssignableFrom(detailsType)) {
                                DetailsListControl detailsProps = new DetailsListControl(detailsFolder, workbenchPart.getSite(), detailsInfo);
                                extDetailsItem.setControl(detailsProps);
                            } else {
                                extDetailsItem.setControl(UIUtils.createLabel(detailsFolder, NLS.bind(SessionEditorMessages.viewer_unsupported_details_type_label, detailsType)));
                            }
                        }
                    }
                }

                detailsFolder.setSelection(detailsItem);
                detailsFolder.addSelectionListener(new SelectionAdapter() {
                    @Override
                    public void widgetSelected(SelectionEvent e) {
                        CTabItem item = detailsFolder.getItem(detailsFolder.getSelectionIndex());
                        Object data = item.getData();
                        if (data instanceof DBAServerSessionDetails) {
                            Class<?> detailsType = ((DBAServerSessionDetails) data).getDetailsType();
                            if (DBPObjectWithDescription.class.isAssignableFrom(detailsType)) {
                                StyledText styledText = (StyledText) item.getControl();
                                loadPlainTextDetails((DBAServerSessionDetails) data, styledText);
                            } else {
                                DetailsListControl detailsViewer = (DetailsListControl) item.getControl();
                                detailsViewer.loadData();
                            }
                        }
                    }
                });
            }

            sashMain.setWeights(new int[]{500, 500});
        }

        sashMain.setWeights(new int[]{700, 300});
    }

    private void loadPlainTextDetails(DBAServerSessionDetails data, StyledText styledText) {
        SessionDetailsLoadService loadingService = new SessionDetailsLoadService(data);
        LoadingJob.createService(
            loadingService,
            new ProgressLoaderVisualizer<Collection<DBPObject>>(loadingService, styledText) {
                @Override
                public void completeLoading(Collection<DBPObject> dbpObjects) {
                    StringBuilder text = new StringBuilder();
                    for (DBPObject item : dbpObjects) {
                        if (item instanceof DBPObjectWithDescription) {
                            text.append(((DBPObjectWithDescription) item).getDescription());
                            text.append(GeneralUtils.getDefaultLineSeparator());
                        }
                    }
                    styledText.setText(text.toString());
                }
            }).schedule();
    }

    private void updatePreview() {
        if (previewFolder.getSelectionIndex() == 0) {
            // Show SQL
            detailsItem.setText(SessionEditorMessages.viewer_details_item_session_details);
            updateSQL();
            if (curSession == null) {
                sessionProps.clearProperties();
            } else {
                PropertyCollector propCollector = new PropertyCollector(curSession, true);
                propCollector.collectProperties();
                sessionProps.loadProperties(propCollector);
            }
        } else if (planViewer != null) {
            // Show execution plan
            String sqlText = curSession == null ? "" : CommonUtils.notEmpty(curSession.getActiveQuery());
            if (!CommonUtils.isEmpty(sqlText)) {
                planViewer.explainQueryPlan(new SQLQuery(sessionManager.getDataSource(), sqlText), curSession.getActiveQueryId());
            }
        }
        if (detailsFolder.getSelectionIndex() > 0) {
            CTabItem detailsItem = detailsFolder.getItem(detailsFolder.getSelectionIndex());
            Object data = detailsItem.getData();
            if (data instanceof DBAServerSessionDetails) {
                if (detailsItem.getControl() instanceof StyledText) {
                    StyledText styledText = (StyledText) detailsItem.getControl();
                    loadPlainTextDetails((DBAServerSessionDetails) data, styledText);
                } else {
                    DetailsListControl detailsListControl = (DetailsListControl) detailsItem.getControl();
                    detailsListControl.loadData();
                }
            }
        }
    }

    private void createPlannerTab(CTabFolder previewFolder) {
        planViewer = new ExplainPlanViewer(workbenchPart, sqlViewer, previewFolder, 0);

//        planTree = new PlanNodesTree(previewFolder, SWT.SHEET, workbenchPart.getSite());
//        planTree.substituteProgressPanel(getSessionListControl());
        planViewer.addSelectionChangedListener(event -> showPlanNode());


        CTabItem sqlPlanItem = new CTabItem(previewFolder, SWT.NONE);
        sqlPlanItem.setText(SessionEditorMessages.viewer_sql_plan_item_execution_plan);
        sqlPlanItem.setImage(DBeaverIcons.getImage(UIIcon.SQL_PAGE_EXPLAIN_PLAN));
        sqlPlanItem.setControl(planViewer.getControl());
        sqlPlanItem.setData(planViewer);
    }

    private void showPlanNode()
    {
        detailsItem.setText(SessionEditorMessages.viewer_plan_details_text);

        ISelection selection = planViewer.getSelection();
        if (selection.isEmpty()) {
            sessionProps.clearProperties();
        } else if (selection instanceof IStructuredSelection) {
            Object element = ((IStructuredSelection) selection).getFirstElement();
            if (element != selectedPlanElement) {
                PropertyCollector propertySource = new PropertyCollector(element, true);
                propertySource.collectProperties();
                sessionProps.loadProperties(propertySource);
                selectedPlanElement = element;
            }
        }
    }

    public DatabaseObjectListControl getSessionListControl() {
        return sessionTable;
    }

    public void dispose()
    {
        sessionTable.disposeControl();
        UIUtils.dispose(boldFont);
    }

    protected void onSessionSelect(DBAServerSession session)
    {
        if (curSession == session && selectedPlanElement == null) {
            return;
        }
        selectedPlanElement = null;
        previewFolder.setSelection(0);
        curSession = session;
        updatePreview();
    }

    protected void contributeToToolbar(DBAServerSessionManager sessionManager, IContributionManager contributionManager)
    {

    }

    @Nullable
    public DBAServerSession getSelectedSession() {
        return sessionTable.getSuitableSelectedElement(DBAServerSession.class);
    }

    public List<DBAServerSession> getSelectedSessions()
    {
        ISelection selection = sessionTable.getSelectionProvider().getSelection();
        if (selection instanceof IStructuredSelection && !selection.isEmpty()) {
            List<DBAServerSession> sessions = new ArrayList<>();
            for (Object item : ((IStructuredSelection) selection).toArray()) {
                if (item instanceof DBAServerSession) {
                    sessions.add((DBAServerSession) item);
                }
            }
            return sessions;
        } else {
            return Collections.emptyList();
        }
    }

    public void refreshSessions() {
        sessionTable.loadData();
        refreshControl.scheduleAutoRefresh(false);
    }

    public void alterSessions(final List<SESSION_TYPE> sessions, Map<String, Object> options) {
        sessionTable.createAlterService(sessions, options).schedule();
    }

    private void updateSQL() {
        String text = curSession == null ? "" : CommonUtils.notEmpty(curSession.getActiveQuery());
        StringEditorInput sqlInput = new StringEditorInput(sessionTable.getShell().getText(), text, true, GeneralUtils.getDefaultFileEncoding());
        if (sqlViewer.getSite() == null) {
            try {
                sqlViewer.init(subSite, sqlInput);
            } catch (PartInitException e) {
                DBWorkbench.getPlatformUI().showError(sessionTable.getShell().getText(), null, e);
            }
        } else {
            sqlViewer.setInput(sqlInput);
        }
        if (sqlViewer.getTextViewer() != null) {
            sqlViewer.reloadSyntaxRules();
        }
    }

    public Composite getControl() {
        return sessionTable.getControl();
    }

    public Map<String, Object> getSessionOptions() {
        return null;
    }

    void loadSettings(AbstractSessionEditor sessionEditor) {
        //$NON-NLS-1$
        settings = UIUtils.getDialogSettings("DBeaver." + sessionEditor.getClass().getSimpleName());
        loadSettings(settings);
    }

    protected void loadSettings(IDialogSettings settings) {
        int mainSashRatio = CommonUtils.toInt(settings.get("MainSashRatio"), 0);
        if (mainSashRatio > 0) {
            sashMain.setWeights(new int[] { mainSashRatio, 1000 - mainSashRatio });
        }
        int detailsSashRatio = CommonUtils.toInt(settings.get("DetailsSashRatio"), 0);
        if (detailsSashRatio > 0) {
            sashDetails.setWeights(new int[] { detailsSashRatio, 1000 - detailsSashRatio });
        }
    }

    protected void saveSettings(IDialogSettings settings) {
        settings.put("MainSashRatio", sashMain.getWeights()[0]);
        settings.put("DetailsSashRatio", sashDetails.getWeights()[0]);
    }

    protected boolean sessionMatches(SESSION_TYPE element, Pattern searchPattern) {
        String activeQuery = element.getActiveQuery();
        if (activeQuery != null && searchPattern.matcher(activeQuery).find()) {
            return true;
        }
        return false;
    }

    private class SessionListControl extends SessionTable<SESSION_TYPE> {

        private final SessionSearcher searcher;

        SessionListControl(Composite sash, IWorkbenchSite site, DBAServerSessionManager<SESSION_TYPE> sessionManager)
        {
            super(sash, SWT.SHEET, site, sessionManager);
            searcher = new SessionSearcher();
        }

        @Override
        public void fillCustomActions(IContributionManager contributionManager) {
            contributeToToolbar(getSessionManager(), contributionManager);

            if (sessionManager instanceof DBAServerSessionManagerSQL &&
                ((DBAServerSessionManagerSQL) sessionManager).canGenerateSessionReadQuery())
            {
                contributionManager.add(ActionUtils.makeActionContribution(new Action(SessionEditorMessages.viewer_open_sql_editor_text, IAction.AS_PUSH_BUTTON) {
                    {
                        setImageDescriptor(DBeaverIcons.getImageDescriptor(DBIcon.TREE_SCRIPT));
                        setToolTipText(SessionEditorMessages.viewer_open_sql_editor_tip);
                    }
                    @Override
                    public void run()
                    {
                        String sqlScript = ((DBAServerSessionManagerSQL) sessionManager).generateSessionReadQuery(getSessionOptions());
                        if (!CommonUtils.isEmpty(sqlScript)) {
                            SQLNavigatorContext navContext = new SQLNavigatorContext(
                                sessionManager.getDataSource().getContainer(),
                                DBUtils.getDefaultContext(sessionManager.getDataSource(), false)
                            );
                            try {
                                SQLEditorHandlerOpenObjectConsole.openAndExecuteSQLScript(
                                    UIUtils.getActiveWorkbenchWindow(),
                                    navContext,
                                    SessionEditorMessages.viewer_session_manager_sql_title,
                                    true,
                                    new StructuredSelection(),
                                    sqlScript
                                );
                            } catch (CoreException e) {
                                DBWorkbench.getPlatformUI().showError(SessionEditorMessages.viewer_session_manager_error_opening_editor_title,
                                    SessionEditorMessages.viewer_session_manager_error_opening_editor_message, e);
                            }
                        }
                    }
                }, true));
                contributionManager.add(new Separator());
            }

            refreshControl.populateRefreshButton(contributionManager);
            contributionManager.add(new Action(SessionEditorMessages.viewer_session_manager_refresh_sessions_action, DBeaverIcons.getImageDescriptor(UIIcon.REFRESH)) {
                @Override
                public void run()
                {
                    refreshSessions();
                }
            });
        }

        @Override
        protected final Map<String, Object> getSessionOptions() {
            return SessionManagerViewer.this.getSessionOptions();
        }

        @Override
        protected ISearchExecutor getSearchRunner()
        {
            return searcher;
        }

        private class SessionSearcher implements ISearchExecutor {

            @Override
            public boolean performSearch(String searchString, int options) {
                try {
                    SearchFilter searchFilter = new SearchFilter(
                        searchString,
                        (options & SEARCH_CASE_SENSITIVE) != 0);
                    getItemsViewer().setFilters(searchFilter);
                    return true;
                } catch (PatternSyntaxException e) {
                    log.error(e.getMessage());
                    return false;
                }
            }

            @Override
            public void cancelSearch() {
                getItemsViewer().setFilters();
            }
        }

        @Override
        protected LoadingJob<Collection<SESSION_TYPE>> createLoadService(boolean forUpdate) {
            return LoadingJob.createService(new LoadSessionsService(), new SessionLoadVisualizer());
        }

        private class SearchFilter extends ViewerFilter {
            final Pattern pattern;

            public SearchFilter(String searchString, boolean caseSensitiveSearch) throws PatternSyntaxException {
                pattern = Pattern.compile(SQLUtils.makeLikePattern(searchString), caseSensitiveSearch ? 0 : Pattern.CASE_INSENSITIVE);
            }

            @Override
            public boolean select(Viewer viewer, Object parentElement, Object element) {
                try {
                    if (element instanceof DBAServerSession) {
                        boolean matches = false;
                        for (DBPPropertyDescriptor property : getAllProperties()) {
                            if (property instanceof ObjectPropertyDescriptor) {
                                Object value = ((ObjectPropertyDescriptor) property).readValue(element, null, true);
                                if (value != null && pattern.matcher(CommonUtils.toString(value)).find()) {
                                    matches = true;
                                    break;
                                }
                            }
                        }
                        return matches;
                    }
                    return false;
                } catch (Exception e) {
                    log.error(e);
                    return false;
                }
            }
        }

        private final class SessionLoadVisualizer extends ObjectsLoadVisualizer {
            @Override
            public void completeLoading(Collection<SESSION_TYPE> items) {
                Collection<DBAServerSession> previouslySelectedSessions = getSelectedSessions();
                super.completeLoading(items);
                if (items != null) {
                    Object[] sessionsToSelect = previouslySelectedSessions.stream().filter(items::contains).toArray();
                    sessionTable.getItemsViewer().setSelection(new StructuredSelection(sessionsToSelect));
                    if (items.contains(curSession)) {
                        onSessionSelect(curSession);
                    } else {
                        onSessionSelect(null);
                    }
                }
            }

            @Override
            protected void afterCompleteLoading(@NotNull Collection<SESSION_TYPE> items) {
                setListData(items, false, false, true);
            }
        }
    }

    private class DetailsListControl extends DatabaseObjectListControl<DBPObject> {

        private DBAServerSessionDetails sessionDetails;

        DetailsListControl(Composite parent, IWorkbenchSite site, DBAServerSessionDetails sessionDetails) {
            super(parent, SWT.SHEET, site, new ListContentProvider());
            this.sessionDetails = sessionDetails;
        }

        @NotNull
        @Override
        protected String getListConfigId(List<Class<?>> classList) {
            return "SessionDetails/" + sessionManager.getDataSource().getContainer().getDriver().getId() + "/" + sessionDetails.getDetailsTitle();
        }

        @Override
        protected Class<?>[] getListBaseTypes(Collection<DBPObject> items) {
            return new Class[] { sessionDetails.getDetailsType() };
        }

        @Override
        protected LoadingJob<Collection<DBPObject>> createLoadService(boolean forUpdate) {
            return LoadingJob.createService(
                new SessionDetailsLoadService(sessionDetails),
                new ObjectsLoadVisualizer());
        }
    }

    private class SessionDetailsLoadService extends DatabaseLoadService<Collection<DBPObject>> {

        private DBAServerSessionDetails sessionDetails;

        SessionDetailsLoadService(DBAServerSessionDetails sessionDetails) {
            super("Load session details " + sessionDetails.getDetailsTitle(), sessionManager.getDataSource());
            this.sessionDetails = sessionDetails;
        }

        @Override
        public Collection<DBPObject> evaluate(DBRProgressMonitor monitor)
            throws InvocationTargetException, InterruptedException
        {
            if (curSession == null) {
                return Collections.emptyList();
            }
            try {
                DBCExecutionContext context = DBUtils.getDefaultContext(sessionManager.getDataSource(), false);
                try (DBCSession session = context.openSession(monitor, DBCExecutionPurpose.UTIL, "Load session details (" + sessionDetails.getDetailsTitle() + ")")) {
                    List<? extends DBPObject> sessionDetails = this.sessionDetails.getSessionDetails(session, curSession);
                    List<DBPObject> result = new ArrayList<>();
                    if (sessionDetails != null) {
                        result.addAll(sessionDetails);
                    }
                    return result;
                }
            } catch (Throwable ex) {
                throw new InvocationTargetException(ex);
            }
        }

    }
}