/*
 * DBeaver - Universal Database Manager
 * Copyright (C) 2010-2023 DBeaver Corp and others
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jkiss.dbeaver.ui.editors.content;

import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.*;
import org.eclipse.ui.editors.text.IEncodingSupport;
import org.jkiss.code.NotNull;
import org.jkiss.code.Nullable;
import org.jkiss.dbeaver.DBException;
import org.jkiss.dbeaver.Log;
import org.jkiss.dbeaver.model.DBIcon;
import org.jkiss.dbeaver.model.DBPContextProvider;
import org.jkiss.dbeaver.model.DBValueFormatting;
import org.jkiss.dbeaver.model.data.DBDContent;
import org.jkiss.dbeaver.model.data.DBDContentCached;
import org.jkiss.dbeaver.model.data.DBDContentStorage;
import org.jkiss.dbeaver.model.data.DBDContentStorageLocal;
import org.jkiss.dbeaver.model.data.storage.BytesContentStorage;
import org.jkiss.dbeaver.model.data.storage.ExternalContentStorage;
import org.jkiss.dbeaver.model.data.storage.StringContentStorage;
import org.jkiss.dbeaver.model.data.storage.TemporaryContentStorage;
import org.jkiss.dbeaver.model.exec.DBCException;
import org.jkiss.dbeaver.model.exec.DBCExecutionContext;
import org.jkiss.dbeaver.model.runtime.DBRProgressMonitor;
import org.jkiss.dbeaver.model.runtime.DefaultProgressMonitor;
import org.jkiss.dbeaver.runtime.DBWorkbench;
import org.jkiss.dbeaver.runtime.LocalFileStorage;
import org.jkiss.dbeaver.ui.DBeaverIcons;
import org.jkiss.dbeaver.ui.IRefreshablePart;
import org.jkiss.dbeaver.ui.UIUtils;
import org.jkiss.dbeaver.ui.data.IAttributeController;
import org.jkiss.dbeaver.ui.data.IValueController;
import org.jkiss.dbeaver.ui.editors.IStatefulEditorInput;
import org.jkiss.dbeaver.ui.editors.StringEditorInput;
import org.jkiss.dbeaver.utils.ContentUtils;
import org.jkiss.dbeaver.utils.GeneralUtils;
import org.jkiss.dbeaver.utils.RuntimeUtils;
import org.jkiss.utils.CommonUtils;
import org.jkiss.utils.IOUtils;

import java.io.*;

/**
 * ContentEditorInput
 */
public class ContentEditorInput implements IPathEditorInput, IStatefulEditorInput, DBPContextProvider, IEncodingSupport
{
    private static final Log log = Log.getLog(ContentEditorInput.class);

    private IValueController valueController;
    private IEditorPart[] editorParts;
    private IEditorPart defaultPart;

    private boolean contentDetached = false;
    private File contentFile;
    private String fileCharset;
    private StringEditorInput.StringStorage stringStorage;

    public ContentEditorInput(
        @NotNull IValueController valueController,
        @Nullable IEditorPart[] editorParts,
        @Nullable IEditorPart defaultPart,
        @NotNull DBRProgressMonitor monitor)
        throws DBException
    {
        this.valueController = valueController;
        this.editorParts = editorParts;
        this.defaultPart = defaultPart;
        this.fileCharset = getDefaultEncoding();
        this.prepareContent(monitor);
    }
    public ContentEditorInput(
        @NotNull IValueController valueController,
        @Nullable IEditorPart[] editorParts,
        @Nullable IEditorPart defaultPart,
        @Nullable String charset,
        @NotNull DBRProgressMonitor monitor)
        throws DBException
    {
        this.valueController = valueController;
        this.editorParts = editorParts;
        this.defaultPart = defaultPart;
        this.fileCharset = CommonUtils.isEmpty(charset) ? getDefaultEncoding() : charset;
        this.prepareContent(monitor);
    }

    public IValueController getValueController()
    {
        return valueController;
    }

    public void refreshContent(DBRProgressMonitor monitor, IValueController valueController) throws DBException
    {
        this.valueController = valueController;
        this.prepareContent(monitor);
    }

    IEditorPart[] getEditors()
    {
        return editorParts;
    }

    public IEditorPart getDefaultEditor() {
        return defaultPart;
    }

    @Override
    public boolean exists()
    {
        return false;
    }

    @Override
    public ImageDescriptor getImageDescriptor()
    {
        return DBeaverIcons.getImageDescriptor(DBIcon.TYPE_LOB);
    }

    @Override
    public String getName()
    {
        String inputName;
        if (valueController instanceof IAttributeController) {
            inputName = ((IAttributeController) valueController).getColumnId();
        } else {
            inputName = valueController.getValueName();
        }
        if (isReadOnly()) {
            inputName += " [Read Only]";
        }
        return inputName;
    }

    @Nullable
    @Override
    public IPersistableElement getPersistable()
    {
        return null;
    }

    @Override
    public String getToolTipText()
    {
        return getName();
    }

    @Nullable
    @Override
    public <T> T getAdapter(Class<T> adapter)
    {
        if (adapter == IStorage.class) {
            if (stringStorage != null) {
                return adapter.cast(stringStorage);
            } else {
                return adapter.cast(new LocalFileStorage(contentFile, fileCharset));
            }
        }
        return null;
    }

    public Object getValue() {
        return valueController.getValue();
    }

    public long getContentLength() {
        if (stringStorage != null) {
            return stringStorage.getLength();
        } else if (contentFile != null) {
            return contentFile.length();
        } else {
            return 0;
        }
    }
    private void prepareContent(DBRProgressMonitor monitor)
        throws DBException
    {
        final Object[] value = new Object[1];
        UIUtils.syncExec(() -> value[0] = getValue());
        DBDContent content;
        if (value[0] instanceof DBDContent) {
            content = (DBDContent) value[0];
        } else {
            // No need to do init
            stringStorage = new StringEditorInput(getName(), CommonUtils.toString(value[0]), isReadOnly(), fileCharset).getStorage();
            return;
        }

        DBDContentStorage storage = content.getContents(monitor);

        if (contentDetached) {
            release();
            contentDetached = false;
        }
        if (storage instanceof DBDContentStorageLocal) {
            // User content's storage directly
            contentFile = ((DBDContentStorageLocal)storage).getDataFile().toFile();
            contentDetached = true;
        } else {
            // Copy content to local file
            try {
                // Create file
                if (contentFile == null) {
                    String valueId;
                    if (valueController instanceof IAttributeController) {
                        valueId = ((IAttributeController) valueController).getColumnId();
                    } else {
                        valueId = valueController.getValueName();
                    }

                    contentFile = ContentUtils.createTempContentFile(monitor, DBWorkbench.getPlatform(), valueId).toFile();
                }

                // Write value to file
                copyContentToFile(content, monitor);
            }
            catch (IOException e) {
                // Delete temp file
                if (contentFile != null && contentFile.exists()) {
                    if (!contentFile.delete()) {
                        log.warn("Can't delete temporary content file '" + contentFile.getAbsolutePath() + "'");
                    }
                }
                throw new DBException("Can't delete content file", e);
            }
        }

        // Mark file as readonly
        if (valueController.isReadOnly()) {
            markReadOnly(true);
        }
    }

    private void markReadOnly(boolean readOnly) throws DBException
    {
        if (!contentFile.setWritable(!readOnly)) {
            throw new DBException("Can't set content read-only");
        }
    }

    public void release()
    {
        if (contentFile != null && !contentDetached) {
            if (!contentFile.delete()) {
                log.warn("Can't delete temp file '" + contentFile.getAbsolutePath() + "'");
            }
            contentDetached = true;
        }
        stringStorage = null;
    }

    @Nullable
    @Override
    public IPath getPath()
    {
        return contentFile == null ?
            new Path("fake_path") : // To avoid NPE from the Eclipse
            new Path(contentFile.getAbsolutePath());
    }

    public boolean isReadOnly() {
        return valueController.isReadOnly();
    }

    void saveToExternalFile(File file, IProgressMonitor monitor)
        throws CoreException
    {
        try (InputStream is = openContents()) {
            ContentUtils.saveContentToFile(
                is,
                file,
                RuntimeUtils.makeMonitor(monitor));
        }
        catch (Exception e) {
            throw new CoreException(GeneralUtils.makeExceptionStatus(e));
        }
    }

    private InputStream openContents() throws Exception {
        return stringStorage == null ? new FileInputStream(contentFile) : stringStorage.getContents();
    }

    void loadFromExternalFile(File extFile, IProgressMonitor monitor)
        throws CoreException
    {
        try {
            release();
            contentFile = extFile;
            contentDetached = true;
            Object value = getValue();
            if (value instanceof DBDContent) {
                ((DBDContent)value).updateContents(
                    new DefaultProgressMonitor(monitor),
                    new ExternalContentStorage(DBWorkbench.getPlatform(), extFile.toPath()));
            } else {
                updateStringValueFromFile(extFile);
            }
            refreshContentParts(extFile);
        }
        catch (Throwable e) {
            throw new CoreException(GeneralUtils.makeExceptionStatus(e));
        }
    }

    private void updateStringValueFromFile(File extFile) throws DBException {
        try (FileReader is = new FileReader(extFile)) {
            String str = IOUtils.readToString(is);
            stringStorage.setString(str);
            valueController.updateValue(str, false);

        } catch (IOException e) {
            throw new DBException("Error reading content from file", e);
        }
    }

    void refreshContentParts(Object source) {
        // Refresh editor parts
        for (IEditorPart cePart : editorParts) {
            if (cePart instanceof IRefreshablePart) {
                ((IRefreshablePart) cePart).refreshPart(source, false);
            }
        }
    }

    private void copyContentToFile(DBDContent contents, DBRProgressMonitor monitor)
        throws DBException, IOException
    {
        DBDContentStorage storage = contents.getContents(monitor);

        markReadOnly(false);

        try (OutputStream os = new FileOutputStream(contentFile)) {
            if (contents.isNull()) {
                ContentUtils.copyStreams(new ByteArrayInputStream(new byte[0]), 0, os, monitor);
            } else {
                if (storage == null) {
                    log.warn("Can't get data from null storage");
                    return;
                }
                try (InputStream is = storage.getContentStream()) {
                    ContentUtils.copyStreams(is, storage.getContentLength(), os, monitor);
                }
            }
        }

        markReadOnly(valueController.isReadOnly());
    }

    public void updateContentFromFile(DBRProgressMonitor monitor, Object value)
        throws DBException
    {
        if (valueController.isReadOnly()) {
            throw new DBCException("Can't update read-only value");
        }

        if (value instanceof DBDContent) {
            DBDContent content = (DBDContent) value;
            DBDContentStorage storage = content.getContents(monitor);
            if (storage instanceof DBDContentStorageLocal) {
                // Nothing to update - we use content's storage
                content.updateContents(monitor, storage);
                contentDetached = true;
            } else if (storage instanceof DBDContentCached) {
                // Create new storage and pass it to content
                try (FileInputStream is = new FileInputStream(contentFile)) {
                    if (ContentUtils.isTextContent(content)) {
                        try (Reader reader = new InputStreamReader(is, fileCharset)) {
                            storage = StringContentStorage.createFromReader(reader);
                        }
                    } else {
                        storage = BytesContentStorage.createFromStream(is, contentFile.length(), fileCharset);
                    }
                    //StringContentStorage.
                    contentDetached = content.updateContents(monitor, storage);
                } catch (IOException e) {
                    throw new DBException("Error reading content from file", e);
                }
            } else {
                // Create new storage and pass it to content
                storage = new TemporaryContentStorage(DBWorkbench.getPlatform(), contentFile.toPath(), fileCharset, false);
                contentDetached = content.updateContents(monitor, storage);
            }
        } else if (stringStorage != null) {
            // Just read as string
            valueController.updateValue(stringStorage.getString(), false);
            contentDetached = true;
        }
    }

    @Nullable
    @Override
    public DBCExecutionContext getExecutionContext() {
        return valueController.getExecutionContext();
    }

    public String getEncoding() {
        return fileCharset;
    }

    @Override
    public String getDefaultEncoding() {
        if (valueController.getExecutionContext() == null) {
            return GeneralUtils.getDefaultFileEncoding();
        }
        return DBValueFormatting.getDefaultBinaryFileEncoding(valueController.getExecutionContext().getDataSource());
    }

    public void setEncoding(String fileCharset) {
        this.fileCharset = fileCharset;
        for (IEditorPart part : editorParts) {
            try {
                if (part instanceof IReusableEditor) {
                    ((IReusableEditor) part).setInput(this);
                } else {
                    part.init(part.getEditorSite(), this);
                }
            } catch (PartInitException e) {
                log.error("Error refreshing content editor part " + part, e);
            }
        }
    }
}