/*
 * DBeaver - Universal Database Manager
 * Copyright (C) 2010-2023 DBeaver Corp and others
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jkiss.dbeaver.model.sql.generator;

import org.jkiss.dbeaver.DBException;
import org.jkiss.dbeaver.model.DBPEvaluationContext;
import org.jkiss.dbeaver.model.DBUtils;
import org.jkiss.dbeaver.model.impl.sql.SelectTableGenerator;
import org.jkiss.dbeaver.model.runtime.DBRProgressMonitor;
import org.jkiss.dbeaver.model.struct.DBSEntity;
import org.jkiss.dbeaver.model.struct.DBSEntityAttribute;

public class SQLGeneratorSelect extends SQLGeneratorTable {
    private boolean columnList = true;

    public void setColumnList(boolean columnList) {
        this.columnList = columnList;
    }

    @Override
    public void generateSQL(DBRProgressMonitor monitor, StringBuilder sql, DBSEntity object) throws DBException {
        if (object instanceof SelectTableGenerator) {
            // It can be non relation database, which have another SELECT statement
            ((SelectTableGenerator) object).createSelectStatement(monitor, sql);
            return;
        }
        sql.append("SELECT ");
        boolean hasAttr = false;
        if (columnList) {
            for (DBSEntityAttribute attr : getAllAttributes(monitor, object)) {
                if (DBUtils.isHiddenObject(attr)) {
                    continue;
                }
                if (hasAttr) sql.append(", ");
                sql.append(DBUtils.getObjectFullName(attr, DBPEvaluationContext.DML));
                hasAttr = true;
            }
            if (hasAttr) {
                sql.append(getLineSeparator());
            }
        }
        if (!hasAttr) {
            sql.append("* ");
        }
        sql.append("FROM ").append(getEntityName(object));
        sql.append(";\n");
    }
}
