/*
 * DBeaver - Universal Database Manager
 * Copyright (C) 2010-2023 DBeaver Corp and others
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jkiss.dbeaver.model.impl.jdbc;

import org.jkiss.code.NotNull;
import org.jkiss.code.Nullable;
import org.jkiss.dbeaver.DBException;
import org.jkiss.dbeaver.Log;
import org.jkiss.dbeaver.model.DBPDataSource;
import org.jkiss.dbeaver.model.DBPDataTypeProvider;
import org.jkiss.dbeaver.model.DBPIdentifierCase;
import org.jkiss.dbeaver.model.DBPKeywordType;
import org.jkiss.dbeaver.model.exec.jdbc.JDBCDatabaseMetaData;
import org.jkiss.dbeaver.model.exec.jdbc.JDBCSession;
import org.jkiss.dbeaver.model.impl.sql.BasicSQLDialect;
import org.jkiss.dbeaver.model.sql.SQLConstants;
import org.jkiss.dbeaver.model.sql.SQLDataTypeConverter;
import org.jkiss.dbeaver.model.sql.SQLDialect;
import org.jkiss.dbeaver.model.sql.SQLStateType;
import org.jkiss.dbeaver.model.struct.DBSDataType;
import org.jkiss.dbeaver.model.struct.DBSTypedObject;
import org.jkiss.utils.CommonUtils;

import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.*;

/**
 * SQL Dialect JDBC API implementation
 */
public class JDBCSQLDialect extends BasicSQLDialect implements SQLDataTypeConverter {

    private static final Log log = Log.getLog(JDBCSQLDialect.class);

    private static final String[] LONG_TEXT_TYPES = {
        "longtext",
        "clob",
        "text",
        "string",
        "nclob"
    };

    private final String name;
    private final String id;
    private String[][] identifierQuoteString = new String[][]{{SQLConstants.DEFAULT_IDENTIFIER_QUOTE, SQLConstants.DEFAULT_IDENTIFIER_QUOTE}};
    private SQLStateType sqlStateType;
    private String searchStringEscape;
    private String catalogSeparator = String.valueOf(SQLConstants.STRUCT_SEPARATOR);
    private boolean isCatalogAtStart;
    private int catalogUsage = SQLDialect.USAGE_ALL;
    protected int schemaUsage = SQLDialect.USAGE_ALL;
    protected String validCharacters = "";
    private boolean supportsUnquotedMixedCase;
    private boolean supportsQuotedMixedCase;
    @NotNull
    private DBPIdentifierCase unquotedIdentCase = getDefaultIdentifiersCase();
    @NotNull
    private DBPIdentifierCase quotedIdentCase = getDefaultIdentifiersCase();
    private boolean supportsSubqueries = false;

    private transient boolean typesLoaded = false;

    public JDBCSQLDialect(String name, String id) {
        this.name = name;
        this.id = id;
    }

    public void initDriverSettings(JDBCSession session, JDBCDataSource dataSource, JDBCDatabaseMetaData metaData) {
        String singleQuoteStr;
        try {
            singleQuoteStr = metaData.getIdentifierQuoteString();
        } catch (Throwable e) {
            log.debug("Error getting identifierQuoteString: " + e.getMessage());
            singleQuoteStr = SQLConstants.DEFAULT_IDENTIFIER_QUOTE;
        }
        if (singleQuoteStr != null) {
            singleQuoteStr = singleQuoteStr.trim();
            if (singleQuoteStr.isEmpty()) {
                singleQuoteStr = null;
            }
        }
        if (singleQuoteStr == null) {
            identifierQuoteString = new String[0][];
        } else {
            identifierQuoteString = new String[][]{{singleQuoteStr, singleQuoteStr}};
        }

        try {
            switch (metaData.getSQLStateType()) {
                case DatabaseMetaData.sqlStateXOpen:
                    this.sqlStateType = SQLStateType.XOPEN;
                    break;
                case DatabaseMetaData.sqlStateSQL99:
                    this.sqlStateType = SQLStateType.SQL99;
                    break;
                default:
                    this.sqlStateType = SQLStateType.UNKNOWN;
                    break;
            }
        } catch (Throwable e) {
            log.debug("Error getting sqlStateType: " + e.getMessage());
            this.sqlStateType = SQLStateType.UNKNOWN;
        }

        try {
            supportsSubqueries = metaData.supportsCorrelatedSubqueries() ||
                metaData.supportsSubqueriesInComparisons() ||
                metaData.supportsSubqueriesInExists() ||
                metaData.supportsSubqueriesInIns() ||
                metaData.supportsSubqueriesInQuantifieds();
        } catch (Throwable e) {
            log.debug("Error getting supportsSubqueries: " + e.getMessage());
        }

        try {
            this.supportsUnquotedMixedCase = metaData.supportsMixedCaseIdentifiers();
        } catch (Throwable e) {
            log.debug("Error getting supportsUnquotedMixedCase:" + e.getMessage());
            this.supportsUnquotedMixedCase = false;
        }
        try {
            this.supportsQuotedMixedCase = metaData.supportsMixedCaseQuotedIdentifiers();
        } catch (Throwable e) {
            log.debug("Error getting supportsQuotedMixedCase: " + e.getMessage());
            this.supportsQuotedMixedCase = false;
        }
        try {
            if (metaData.storesUpperCaseIdentifiers()) {
                this.unquotedIdentCase = DBPIdentifierCase.UPPER;
            } else if (metaData.storesLowerCaseIdentifiers()) {
                this.unquotedIdentCase = DBPIdentifierCase.LOWER;
            } else {
                this.unquotedIdentCase = DBPIdentifierCase.MIXED;
            }
        } catch (Throwable e) {
            log.debug("Error getting unquotedIdentCase:" + e.getMessage());
            this.unquotedIdentCase = DBPIdentifierCase.MIXED;
        }
        try {
            if (metaData.storesUpperCaseQuotedIdentifiers()) {
                this.quotedIdentCase = DBPIdentifierCase.UPPER;
            } else if (metaData.storesLowerCaseQuotedIdentifiers()) {
                this.quotedIdentCase = DBPIdentifierCase.LOWER;
            } else {
                this.quotedIdentCase = DBPIdentifierCase.MIXED;
            }
        } catch (Throwable e) {
            log.debug("Error getting quotedIdentCase:" + e.getMessage());
            this.quotedIdentCase = DBPIdentifierCase.MIXED;
        }
        try {
            this.searchStringEscape = metaData.getSearchStringEscape();
        } catch (Throwable e) {
            log.debug("Error getting searchStringEscape:" + e.getMessage());
        }
        if (this.searchStringEscape == null) {
            this.searchStringEscape = ""; //$NON-NLS-1$
        }
        try {
            this.catalogSeparator = metaData.getCatalogSeparator();
            if (CommonUtils.isEmpty(this.catalogSeparator)) {
                this.catalogSeparator = String.valueOf(SQLConstants.STRUCT_SEPARATOR);
            }
        } catch (Throwable e) {
            log.debug("Error getting catalogSeparator:" + e.getMessage());
            this.catalogSeparator = String.valueOf(SQLConstants.STRUCT_SEPARATOR);
        }
        try {
            catalogUsage =
                (metaData.supportsCatalogsInDataManipulation() ? SQLDialect.USAGE_DML : 0) |
                    (metaData.supportsCatalogsInTableDefinitions() ? SQLDialect.USAGE_DDL : 0) |
                    (metaData.supportsCatalogsInProcedureCalls() ? SQLDialect.USAGE_PROC : 0) |
                    (metaData.supportsCatalogsInIndexDefinitions() ? SQLDialect.USAGE_INDEX : 0) |
                    (metaData.supportsCatalogsInPrivilegeDefinitions() ? SQLDialect.USAGE_PRIV : 0);
        } catch (Throwable e) {
            log.debug("Error getting catalogUsage:" + e.getMessage());
            catalogUsage = SQLDialect.USAGE_NONE;
        }
        try {
            schemaUsage =
                (metaData.supportsSchemasInDataManipulation() ? SQLDialect.USAGE_DML : 0) |
                    (metaData.supportsSchemasInTableDefinitions() ? SQLDialect.USAGE_DDL : 0) |
                    (metaData.supportsSchemasInProcedureCalls() ? SQLDialect.USAGE_PROC : 0) |
                    (metaData.supportsSchemasInIndexDefinitions() ? SQLDialect.USAGE_INDEX : 0) |
                    (metaData.supportsSchemasInPrivilegeDefinitions() ? SQLDialect.USAGE_PRIV : 0);
        } catch (Throwable e) {
            log.debug("Error getting schemaUsage:" + e.getMessage());
            schemaUsage = SQLDialect.USAGE_DDL | SQLDialect.USAGE_DML;
        }
        try {
            validCharacters = metaData.getExtraNameCharacters();
            if (validCharacters == null) {
                validCharacters = "";
            } else {
                validCharacters = validCharacters.trim();
            }
        } catch (Throwable e) {
            log.debug("Error getting validCharacters:" + e.getMessage());
            validCharacters = ""; //$NON-NLS-1$
        }

        try {
            this.isCatalogAtStart = metaData.isCatalogAtStart();
        } catch (Throwable e) {
            log.debug("Error getting isCatalogAtStart:" + e.getMessage());
            this.isCatalogAtStart = true;
        }

        loadDriverKeywords(session, dataSource, metaData);
    }


    @NotNull
    protected DBPIdentifierCase getDefaultIdentifiersCase() {
        return DBPIdentifierCase.MIXED;
    }

    @NotNull
    @Override
    public String getDialectName() {
        return name;
    }

    @NotNull
    @Override
    public String getDialectId() {
        return id;
    }

    @Nullable
    @Override
    public String[][] getIdentifierQuoteStrings() {
        return identifierQuoteString;
    }

    protected void setIdentifierQuoteString(String[][] identifierQuoteString) {
        this.identifierQuoteString = identifierQuoteString;
    }

    @NotNull
    @Override
    public String[] getExecuteKeywords() {
        return new String[0];
    }

    @NotNull
    @Override
    public String getSearchStringEscape() {
        return searchStringEscape;
    }

    @Override
    public int getCatalogUsage() {
        return catalogUsage;
    }

    @Override
    public int getSchemaUsage() {
        return schemaUsage;
    }

    @NotNull
    @Override
    public String getCatalogSeparator() {
        return catalogSeparator;
    }

    @Override
    public char getStructSeparator() {
        return SQLConstants.STRUCT_SEPARATOR;
    }

    @Override
    public boolean isCatalogAtStart() {
        return isCatalogAtStart;
    }

    @NotNull
    @Override
    public SQLStateType getSQLStateType() {
        return sqlStateType;
    }

    @Override
    public boolean validIdentifierPart(char c, boolean quoted) {
        // Some driver return extra characters which must be quoted. We can't know this here.
        return Character.isLetter(c) || Character.isDigit(c) || c == '_' || (quoted && validCharacters.indexOf(c) != -1);
    }

    @Override
    public boolean supportsUnquotedMixedCase() {
        return supportsUnquotedMixedCase;
    }

    public void setSupportsUnquotedMixedCase(boolean supportsUnquotedMixedCase) {
        this.supportsUnquotedMixedCase = supportsUnquotedMixedCase;
    }

    @Override
    public boolean supportsQuotedMixedCase() {
        return supportsQuotedMixedCase;
    }

    protected void setSupportsQuotedMixedCase(boolean supportsQuotedMixedCase) {
        this.supportsQuotedMixedCase = supportsQuotedMixedCase;
    }

    @NotNull
    @Override
    public DBPIdentifierCase storesUnquotedCase() {
        return unquotedIdentCase;
    }

    protected void setUnquotedIdentCase(@NotNull DBPIdentifierCase unquotedIdentCase) {
        this.unquotedIdentCase = unquotedIdentCase;
    }

    @NotNull
    @Override
    public DBPIdentifierCase storesQuotedCase() {
        return quotedIdentCase;
    }

    @Override
    public boolean supportsSubqueries() {
        return supportsSubqueries;
    }

    public void setSupportsSubqueries(boolean supportsSubqueries) {
        this.supportsSubqueries = supportsSubqueries;
    }

    public boolean supportsUpsertStatement() {
        return false;
    }

    @NotNull
    @Override
    public Collection<String> getDataTypes(@Nullable DBPDataSource dataSource) {
        if (!typesLoaded && dataSource instanceof JDBCDataSource) {
            loadDataTypesFromDatabase((JDBCDataSource) dataSource);
            typesLoaded = true;
        }
        return super.getDataTypes(dataSource);
    }

    protected void loadDataTypesFromDatabase(JDBCDataSource dataSource) {
        clearDataTypes();

        Collection<? extends DBSDataType> supportedDataTypes = dataSource.getLocalDataTypes();
        List<String> dataTypes = new ArrayList<>();
        if (supportedDataTypes != null) {
            for (DBSDataType dataType : supportedDataTypes) {
                if (!dataType.getDataKind().isComplex()) {
                    dataTypes.add(dataType.getName().toUpperCase(Locale.ENGLISH));
                }
            }
        }

        if (dataTypes.isEmpty() && needsDefaultDataTypes()) {
            // Add default types
            Collections.addAll(dataTypes, SQLConstants.DEFAULT_TYPES);
        }
        addDataTypes(dataTypes);
    }

    private void loadDriverKeywords(JDBCSession session, JDBCDataSource dataSource, JDBCDatabaseMetaData metaData) {
        try {
            // Keywords
            Collection<String> sqlKeywords = makeStringList(metaData.getSQLKeywords());
            if (!CommonUtils.isEmpty(sqlKeywords)) {
                for (String keyword : sqlKeywords) {
                    addSQLKeyword(keyword.toUpperCase());
                }
            }
        } catch (SQLException e) {
            log.debug("Error reading SQL keywords: " + e.getMessage());
        }
        try {
            // Functions
            Set<String> allFunctions = new HashSet<>();
            loadFunctions(session, metaData, allFunctions);
            // Remove functions which clashes with keywords
            allFunctions.removeIf(s -> getKeywordType(s) == DBPKeywordType.KEYWORD);
            addFunctions(allFunctions);
        } catch (Throwable e) {
            log.debug("Error reading SQL functions: " + e.getMessage());
        }
    }

    protected void loadFunctions(JDBCSession session, JDBCDatabaseMetaData metaData, Set<String> allFunctions) throws DBException, SQLException {
        allFunctions.addAll(makeStringList(metaData.getNumericFunctions()));
        allFunctions.addAll(makeStringList(metaData.getStringFunctions()));
        allFunctions.addAll(makeStringList(metaData.getSystemFunctions()));
        allFunctions.addAll(makeStringList(metaData.getTimeDateFunctions()));
    }

    private static List<String> makeStringList(String source) {
        List<String> result = new ArrayList<>();
        if (source != null && source.length() > 0) {
            StringTokenizer st = new StringTokenizer(source, ";,"); //$NON-NLS-1$
            while (st.hasMoreTokens()) {
                result.add(st.nextToken().trim());
            }
        }
        return result;
    }

    @Override
    public String convertExternalDataType(@NotNull SQLDialect sourceDialect, @NotNull DBSTypedObject sourceTypedObject, @Nullable DBPDataTypeProvider targetTypeProvider) {
        if (targetTypeProvider != null) {
            String externalTypeName = sourceTypedObject.getTypeName().toLowerCase(Locale.ENGLISH);
            if (SQLConstants.DATA_TYPE_VARCHAR.equals(externalTypeName)) {
                long maxLength = sourceTypedObject.getMaxLength();
                if (maxLength <= 0) {
                    // Some databases can not have varchar data type without modifiers.
                    // Like PostgreSQL where varchar without modifiers == text.
                    // But other databases are more strict in this case
                    // Let's use another text data type instead
                    for (String textType : LONG_TEXT_TYPES) {
                        DBSDataType textDataType = targetTypeProvider.getLocalDataType(textType);
                        if (textDataType != null) {
                            return textDataType.getName();
                        }
                    }
                }
            }
        }
        return null;
    }
}
