/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.core.hint;

import org.apache.shardingsphere.api.hint.HintManager;
import org.junit.After;
import org.junit.Test;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

public final class HintManagerHolderTest {
    
    private final HintManager hintManager = HintManager.getInstance();
    
    @After
    public void tearDown() {
        hintManager.close();
    }
    
    @Test(expected = IllegalStateException.class)
    public void assertSetHintManagerTwice() {
        HintManagerHolder.setHintManager(HintManager.getInstance());
    }
    
    @Test
    public void assertGetDatabaseShardingValuesWithoutLogicTable() {
        hintManager.setDatabaseShardingValue(1);
        assertThat(HintManagerHolder.getDatabaseShardingValues().size(), is(1));
        assertTrue(HintManagerHolder.getDatabaseShardingValues().contains(1));
    }
    
    @Test
    public void assertGetDatabaseShardingValuesWithLogicTable() {
        hintManager.addDatabaseShardingValue("logic_table", 1);
        assertThat(HintManagerHolder.getDatabaseShardingValues("logic_table").size(), is(1));
        assertTrue(HintManagerHolder.getDatabaseShardingValues("logic_table").contains(1));
    }
    
    @Test
    public void assertGetTableShardingValues() {
        hintManager.addTableShardingValue("logic_table", 1);
        assertThat(HintManagerHolder.getTableShardingValues("logic_table").size(), is(1));
        assertTrue(HintManagerHolder.getTableShardingValues("logic_table").contains(1));
    }
    
    @Test
    public void assertIsDatabaseShardingOnly() {
        hintManager.setDatabaseShardingValue(1);
        assertTrue(HintManagerHolder.isDatabaseShardingOnly());
    }
    
    @Test
    public void assertIsDatabaseShardingOnlyWithoutSet() {
        hintManager.close();
        assertFalse(HintManagerHolder.isDatabaseShardingOnly());
    }
    
    @Test
    public void assertIsMasterRouteOnly() {
        hintManager.setMasterRouteOnly();
        assertTrue(HintManagerHolder.isMasterRouteOnly());
    }
    
    @Test
    public void assertIsMasterRouteOnlyWithoutSet() {
        hintManager.close();
        assertFalse(HintManagerHolder.isMasterRouteOnly());
    }
    
    @Test
    public void assertClear() {
        hintManager.addDatabaseShardingValue("logic_table", 1);
        hintManager.addTableShardingValue("logic_table", 1);
        hintManager.close();
        assertTrue(HintManagerHolder.getDatabaseShardingValues("logic_table").isEmpty());
        assertTrue(HintManagerHolder.getTableShardingValues("logic_table").isEmpty());
    }
}
