import re
import textwrap

import gradio as gr
from bs4 import BeautifulSoup

from modules import chat, shared
from modules.logging_colors import logger

from .chromadb import add_chunks_to_collector, make_collector
from .download_urls import download_urls

params = {
    'chunk_count': 5,
    'chunk_count_initial': 10,
    'time_weight': 0,
    'chunk_length': 700,
    'chunk_separator': '',
    'strong_cleanup': False,
    'threads': 4,
}

collector = make_collector()
chat_collector = make_collector()


def feed_data_into_collector(corpus, chunk_len, chunk_sep):
    global collector

    # Defining variables
    chunk_len = int(chunk_len)
    chunk_sep = chunk_sep.replace(r'\n', '\n')
    cumulative = ''

    # Breaking the data into chunks and adding those to the db
    cumulative += "Разрыв входного датасета...\n\n"
    yield cumulative
    if chunk_sep:
        data_chunks = corpus.split(chunk_sep)
        data_chunks = [[data_chunk[i:i + chunk_len] for i in range(0, len(data_chunk), chunk_len)] for data_chunk in data_chunks]
        data_chunks = [x for y in data_chunks for x in y]
    else:
        data_chunks = [corpus[i:i + chunk_len] for i in range(0, len(corpus), chunk_len)]

    cumulative += f"{len(data_chunks)} фрагменты найдены.\n\nДобавление фрагментов в базу данных...\n\n"
    yield cumulative
    add_chunks_to_collector(data_chunks, collector)
    cumulative += "Готово."
    yield cumulative


def feed_file_into_collector(file, chunk_len, chunk_sep):
    yield 'Чтение входного датасета...\n\n'
    text = file.decode('utf-8')
    for i in feed_data_into_collector(text, chunk_len, chunk_sep):
        yield i


def feed_url_into_collector(urls, chunk_len, chunk_sep, strong_cleanup, threads):
    all_text = ''
    cumulative = ''

    urls = urls.strip().split('\n')
    cumulative += f'Загрузка {len(urls)} URL-адресов с {threads} потоками...\n\n'
    yield cumulative
    for update, contents in download_urls(urls, threads=threads):
        yield cumulative + update

    cumulative += 'Обработка исходников HTML...'
    yield cumulative
    for content in contents:
        soup = BeautifulSoup(content, features="lxml")
        for script in soup(["script", "style"]):
            script.extract()

        strings = soup.stripped_strings
        if strong_cleanup:
            strings = [s for s in strings if re.search("[A-Za-z] ", s)]

        text = '\n'.join([s.strip() for s in strings])
        all_text += text

    for i in feed_data_into_collector(all_text, chunk_len, chunk_sep):
        yield i


def apply_settings(chunk_count, chunk_count_initial, time_weight):
    global params
    params['chunk_count'] = int(chunk_count)
    params['chunk_count_initial'] = int(chunk_count_initial)
    params['time_weight'] = time_weight
    settings_to_display = {k: params[k] for k in params if k in ['chunk_count', 'chunk_count_initial', 'time_weight']}
    yield f"Сейчас активны следующие настройки: {str(settings_to_display)}"


def custom_generate_chat_prompt(user_input, state, **kwargs):
    global chat_collector

    history = state['history']

    if state['mode'] == 'instruct':
        results = collector.get_sorted(user_input, n_results=params['chunk_count'])
        additional_context = '\nВаш ответ должен основываться на приведенном ниже контексте:\n\n' + '\n'.join(results)
        user_input += additional_context
    else:

        def make_single_exchange(id_):
            output = ''
            output += f"{state['name1']}: {history['internal'][id_][0]}\n"
            output += f"{state['name2']}: {history['internal'][id_][1]}\n"
            return output

        if len(history['internal']) > params['chunk_count'] and user_input != '':
            chunks = []
            hist_size = len(history['internal'])
            for i in range(hist_size - 1):
                chunks.append(make_single_exchange(i))

            add_chunks_to_collector(chunks, chat_collector)
            query = '\n'.join(history['internal'][-1] + [user_input])
            try:
                best_ids = chat_collector.get_ids_sorted(query, n_results=params['chunk_count'], n_initial=params['chunk_count_initial'], time_weight=params['time_weight'])
                additional_context = '\n'
                for id_ in best_ids:
                    if history['internal'][id_][0] != '<|BEGIN-VISIBLE-CHAT|>':
                        additional_context += make_single_exchange(id_)

                logger.warning(f'Добавление следующего нового контекста:\n{additional_context}')
                state['context'] = state['context'].strip() + '\n' + additional_context
                kwargs['history'] = {
                    'internal': [history['internal'][i] for i in range(hist_size) if i not in best_ids],
                    'visible': ''
                }
            except RuntimeError:
                logger.error("Не удалось запросить базу данных, идем дальше...")

    return chat.generate_chat_prompt(user_input, state, **kwargs)


def remove_special_tokens(string):
    pattern = r'(<\|begin-user-input\|>|<\|end-user-input\|>|<\|injection-point\|>)'
    return re.sub(pattern, '', string)


def input_modifier(string):
    if shared.is_chat():
        return string

    # Find the user input
    pattern = re.compile(r"<\|begin-user-input\|>(.*?)<\|end-user-input\|>", re.DOTALL)
    match = re.search(pattern, string)
    if match:
        user_input = match.group(1).strip()

        # Get the most similar chunks
        results = collector.get_sorted(user_input, n_results=params['chunk_count'])

        # Make the injection
        string = string.replace('<|injection-point|>', '\n'.join(results))

    return remove_special_tokens(string)


def ui():
    with gr.Accordion("Click for more information...", open=False):
        gr.Markdown(textwrap.dedent("""

        ## О расширении

        Это расширение принимает датасет в качестве входных данных, разбивает его на фрагменты и добавляет результат в локальную/автономную базу данных Chroma.

         Затем база данных запрашивается во время логического вывода, чтобы получить выдержки, наиболее близкие к вашему вводу. Идея состоит в том, чтобы создать произвольно большой псевдоконтекст.

         Основная методология была разработана и предоставлена kaiokendev, который работает над улучшением метода в этом репозитории: https://github.com/kaiokendev/superbig.

         ## Ввод данных

         Начните с ввода некоторых данных в интерфейсе ниже, а затем нажмите «Загрузить данные».

         Каждый раз, когда вы загружаете новые данные, старые фрагменты отбрасываются.

        ## Chat mode

        #### Instruct

        На каждом ходу фрагменты будут сравниваться с вашим текущим вводом, и наиболее релевантные совпадения будут добавляться к вводу в следующем формате:

        ```
        Рассмотрим выдержки ниже в качестве дополнительного контекста:
        ...
        ```

        Инъекция не попадает в историю чата. Он используется только в текущем поколении.

        #### Regular chat

        Фрагменты из внешних источников данных игнорируются, а база данных цветности строится на основе истории чата. Наиболее релевантные прошлые обмены относительно текущего ввода добавляются в строку контекста. Таким образом, расширение действует как долговременная память.

        ## Notebook/default modes

         Ваш вопрос должен быть указан вручную между тегами `<|begin-user-input|>` и `<|end-user-input|>`, а точка ввода должна быть указана с помощью `<|injection-point|>`

        Вот пример в формате Vicuna 1.1:

        ```
        Чат между любопытным пользователем и помощником с искусственным интеллектом. Помощник дает полезные, подробные и вежливые ответы на вопросы пользователя.

        USER:

        <|begin-user-input|>
        Какие датасеты упоминаются в тексте ниже?
        <|end-user-input|>

        <|injection-point|>

        ASSISTANT:
        ```

        ⚠️  Для достижения наилучших результатов не забудьте удалить пробелы и символы новой строки после `ASSISTANT:`.

        *Это расширение в настоящее время является экспериментальным и находится в стадии разработки.*

        """))

    with gr.Row():
        with gr.Column(min_width=600):
            with gr.Tab("Ввод текста"):
                data_input = gr.Textbox(lines=20, label='Входные данные')
                update_data = gr.Button('Load data')

            with gr.Tab("Ввод URL"):
                url_input = gr.Textbox(lines=10, label='Введите URL', info='Введите один или несколько URL-адресов, разделенных символами новой строки.')
                strong_cleanup = gr.Checkbox(value=params['strong_cleanup'], label='Сильная очистка', info='Сохраняет только html-элементы, которые выглядят как длинный текст.')
                threads = gr.Number(value=params['threads'], label='Потоки', info='Количество потоков, используемых при загрузке URL-адресов.', precision=0)
                update_url = gr.Button('Загрузить данные')

            with gr.Tab("Ввод файла"):
                file_input = gr.File(label='Входной файл', type='binary')
                update_file = gr.Button('Загрузить данные')

            with gr.Tab("Настройки генерации"):
                chunk_count = gr.Number(value=params['chunk_count'], label='Количество фрагментов', info='Количество наиболее подходящих фрагментов для включения в промт.')
                gr.Markdown('Взвешивание по времени (необязательно, используется для повышения вероятности появления недавно добавленных фрагментов)')
                time_weight = gr.Slider(0, 1, value=params['time_weight'], label='Time weight', info='Определяет силу взвешивания по времени. 0 = без взвешивания по времени.')
                chunk_count_initial = gr.Number(value=params['chunk_count_initial'], label='Начальное количество фрагментов', info='Количество наиболее подходящих фрагментов, извлеченных для переупорядочивания временного веса в режиме чата. Это должно быть >= количество фрагментов. -1 = все фрагменты извлечены. Используется, только если time_weight > 0.')

                update_settings = gr.Button('Применить изменения')

            chunk_len = gr.Number(value=params['chunk_length'], label='Длина фрагмента', info='В символах, а не в токенах. Это значение используется, когда вы нажимаете "Загрузить данные".')
            chunk_sep = gr.Textbox(value=params['chunk_separator'], label='Разделитель фрагментов', info='Используется для ручного разделения фрагментов. Разделенные вручную фрагменты, длина которых превышает длину фрагмента, снова разделяются. Это значение используется, когда вы нажимаете "Загрузить данные".')
        with gr.Column():
            last_updated = gr.Markdown()

    update_data.click(feed_data_into_collector, [data_input, chunk_len, chunk_sep], last_updated, show_progress=False)
    update_url.click(feed_url_into_collector, [url_input, chunk_len, chunk_sep, strong_cleanup, threads], last_updated, show_progress=False)
    update_file.click(feed_file_into_collector, [file_input, chunk_len, chunk_sep], last_updated, show_progress=False)
    update_settings.click(apply_settings, [chunk_count, chunk_count_initial, time_weight], last_updated, show_progress=False)
