import pickle
import struct
from pathlib import Path
from typing import Union, Any, Set
import random
from uuid import uuid4

class File:
    """
    Cached Additive Data Storage.
    
    Pros:
    
    + Simple
    
    + Fast addition via caching
    
    + Fastest retrieval
    
    + allows Single Writer Multi Reader access (SWMR)
    
    Cons:
    
    - Not suitable for millions of keys.
        
    - Multi writer will corrupt the file.
        
    
        
    ```
    File format
    
    MAGIC NUMBER (4)
    
    VERSION (4)
    
    MODIFIED RND BYTES (4)
        
    Data blocks (sequental)
        
        size of whole block(8)
         
        UUID generated value
    
        Data (sequental)
                    
            pickled value
            
        pickled dict(N) of key:absolute offset of data       
        
        pickled dict size(8)
        
        UUID generated value (same as first)

    ```
    """
    
        
    
    def __init__(self, path : Union[str, Path], create_new=False):
        """```
            path
            
            create_new(False)   if True force recreate a file,
                                otherwise file should exist or error rises.
                                
        raises on errors.
        ```"""
        self._path = path = Path(path)
        
        self._key_offsets = {}
        self._cached_data = {}
        self._mount_data = {}
        
        self._f = None
        
        if create_new:
            f = open(path, 'w+b')
            f.seek(0, 0)
            
            if f.write( struct.pack('II', self._MAGIC_NUMBER, self._VERSION) ) != 8:
                raise IOError()
              
            if f.write( random.randbytes(4) ) != 4:
                raise IOError()
                
            f.flush()
            
            self._new_block_start = f.tell()
            self._f = f
        else:           
            f = open(path, 'r+b')
            f_size = f.seek(0, 2)
            
            if f_size < 8:
                raise self.NotAddyFile()
            
            _, header_data = f.seek(0, 0), f.read(8)            
            if len(header_data) != 8:
                raise self.NotAddyFile()
 
            magic_number, version = struct.unpack('II', header_data)
            if version != self._VERSION or magic_number != self._MAGIC_NUMBER:
                raise self.NotAddyFile()
                
            modified_bytes = f.read(4)
            
            key_offsets = {}
                        
            while True:
                try:
                    new_block_start = f.tell()
                    
                    block_start_offset = f.tell()
                    if block_start_offset >= f_size:
                        break
                    
                    block_size, = struct.unpack('Q', f.read(8))
                    
                    uuid = f.read(16)
                    
                    f.seek(block_start_offset+block_size-16-8, 0)
                    
                    key_offsets_size, = struct.unpack('Q', f.read(8) )
                    
                    if uuid != f.read(16):
                        raise
                    
                    f.seek(block_start_offset+block_size-16-8-key_offsets_size, 0)
                    
                    key_offsets.update(pickle.load(f))
                    
                    f.seek(block_start_offset+block_size, 0)
                    
                except:
                    print(f'{path} has corrupted block at offset ', block_start_offset)
                    break
                
            self._new_block_start = new_block_start
            self._modified_bytes = modified_bytes
            self._key_offsets = key_offsets
            self._f = f
            
    
    def __del__(self):
        if self._f is not None:
            print('WARNING: caddys.File was not closed.')
        
    def is_closed(self) -> bool: return self._f is None

    def is_modified(self) -> bool:
        """
        Check if file was modified since initialization
        
        raises on errors
        """
        if (f := self._f) is None:
            raise self.ClosedFile()
        
        f.seek(8, 0)
        return self._modified_bytes != f.read(4)
        
    def close(self):
        if self._f is not None:
            try:
                self._flush_cache()
                self._f.close()
            except:
                ...
            self._f = None
            self._key_offsets = None
            self._cached_data = None
            self._mount_data = None
            
    def flush(self):
        """
        Flush cached data and file object.
        
        raises on errors
        """
        if self._f is None:
            raise self.ClosedFile()
        
        try:
            self._flush_cache()
            self._f.flush()
        except Exception as e:
            self.close()
            raise e
                       
    def get(self, key, default_value = None) -> Any:
        """```
        Get data of key.
        
        raises on errors
        ```"""
        if (f := self._f) is None:
            raise self.ClosedFile()
        
        if key in self._cached_data:
            return self._cached_data[key]
        else:
            data_offset = self._key_offsets.get(key, None)
            if data_offset is not None:
                try:
                    f.seek(data_offset, 0)
                    return pickle.load(f)
                except Exception as e:
                    self.close()
                    raise e
                
        return default_value
    
    def collect(self, key, value):
        """```
        Collect picklable data of picklable key.
        Existing data will be replaced.
        To write collected data to disk use .flush()
        ```"""        
        self._cached_data[key] = value
        
    def mount(self, key, value):
        """```
        Mount mutable value (for example dict) of key.
        This key-value will be collected before every .flush() 
        
        Same as
        
        .collect(key, value)
        .flush()        
        ```"""
        self._mount_data[key] = value
    
    def unmount(self, key):
        try:
            self._mount_data.pop(key)
        except:
            ...
        
    def keys(self) -> Set[Any]:
        """```Get a set of all keys.```"""
        return set(self._key_offsets.keys()).union(self._cached_data.keys())
    
    def shrink_to(self, path : Path):
        """Build defragmented version of File to path"""
        
        ds = File(path, create_new=True)
        for key in self.keys():
            value = self.get(key)
            ds.collect(key, value)
            
        ds.close()
    
    def _flush_cache(self):
        self._cached_data.update(self._mount_data)
        
        if len(self._cached_data) != 0:
            
            f = self._f
            
            block_start = f.seek(self._new_block_start, 0)
            
            uuid = uuid4()
            
            f.write( bytes(8) ) # Size of block
            f.write( uuid.bytes )
            
            key_offsets = {}
            for key, data in self._cached_data.items():
                key_offsets[key] = f.tell()   
                pickle.dump(data, f)  
                
            self._key_offsets.update(key_offsets)                             
            
            key_offsets_start_offset = f.tell()
            
            pickle.dump(key_offsets, f)    
            
            if f.write( struct.pack('Q', f.tell() - key_offsets_start_offset) ) != 8:
                raise IOError()                
            
            f.write( uuid.bytes )
            
            self._new_block_start = f.tell()
            
            block_size = f.tell() - block_start
            
            f.seek(block_start, 0)
            if f.write( struct.pack('Q', block_size) ) != 8:
                raise IOError()          
                    
            self._cached_data = {}
              
    
    class AddyDSException(Exception):
        def __repr__(self):
            return self.__str__()
        
    class ClosedFile(AddyDSException): 
        def __str__(self):
            return "Closed file."
        
    class NotAddyFile(AddyDSException): 
        def __str__(self):
            return "File is not caddys file.."
                        
    _VERSION = 1
    _MAGIC_NUMBER = 0xADD1ADD1
    