from typing import Callable, Dict, Union

from PySide6.QtCore import *
from PySide6.QtGui import *
from PySide6.QtWidgets import *

from .. import ax, qt
from .QAbstractLiveItemView import QAbstractLiveItemView
from .QPixmap import QPixmap
from .QStyle import QStyle
from .. import time as lib_time

class QLivePixmapView(QAbstractLiveItemView):
    class ItemInfo:
        pixmap : Union[QPixmap, None]
        caption : Union[str, None]
        
    def __init__(self,  task_get_item_info : Callable[ [int], ax.Task[ItemInfo] ] = None,
                    ):
        self._task_get_item_info = task_get_item_info
        self._tg = ax.TaskGroup().dispose_with(self)
        self._tg_cache = ax.TaskGroup().dispose_with(self)

        self._cap_spacing = 1
        self._cap_h = 16                    
        self._cached_ids = set()
        self._cached_ids_task : Dict[int, ax.Task[QLivePixmapView.ItemInfo] ] = {}
        self._cached_ids_item_info : Dict[int, QLivePixmapView.ItemInfo]= {}
        
        super().__init__()
        
        
    def dispose(self):
        super().dispose()
        self._tg.dispose()
        self._tg_cache.dispose()
        self._cached_ids = None
        self._cached_ids_task = None
        self._cached_ids_item_info = None
        
    
    def set_item_size(self, w, h):
        return super().set_item_size(w, h + self._cap_spacing + self._cap_h)
    
    
    @ax.task     
    def _cache_item(self, id):
        task_get_item_info = self._task_get_item_info
        
        if id in self._cached_ids or task_get_item_info is None:
            return
        
        yield ax.attach_to(self._tg)
        yield ax.attach_to(self._tg_cache)
        
        self._cached_ids.add(id)
        
        yield ax.sleep(0)
        
        self._cached_ids_task[id] = task = task_get_item_info(id)
        self._cached_ids_item_info[id] = None
        
        yield ax.wait(task)
        
        if task.succeeded:
            self._cached_ids_item_info[id] = task.result
            
            id_start = self._visible_idxs_start
            id_count = self._visible_idxs_count
        
            if len(self._cached_ids) > id_count*2:
                for id in self._cached_ids:
                    if id < id_start or id >= id_start+id_count:
                        break
                    id = None
                
                if id is not None:                
                    self._cached_ids.remove(id)
                    self._cached_ids_task.pop(id).cancel()
                    self._cached_ids_item_info.pop(id)
            
            self.update()
            
    def clear_cache(self):
        self._tg_cache.cancel_all()                
        self._cached_ids = set()
        self._cached_ids_task = {}
        self._cached_ids_item_info = {}        
        self.update()
        
    def _update_geometry(self,  new_item_w = None,
                                new_item_h = None,
                                new_spacing = None,
                                new_item_count = None,
                                new_view_row = None):
        # Override _update_geometry in order to track new visible items
        # visible_idxs_start = self._visible_idxs_start
        # visible_idxs_count = self._visible_idxs_count
        
        super()._update_geometry(new_item_w = new_item_w,
                                 new_item_h = new_item_h,
                                 new_spacing = new_spacing,
                                 new_item_count = new_item_count,
                                 new_view_row = new_view_row) 
        
        # if (self._visible_idxs_start-visible_idxs_start + self._visible_idxs_count-visible_idxs_count) != 0:
        #     for id in range(self._visible_idxs_start, self._visible_idxs_start+self._visible_idxs_count):
        #         print('cache', id)
                
        #         self._cache_item(id)
                                         
                                 
    def _on_paint_item(self, id : int, qp : QPainter, rect : QRect):
        """overridable"""
          
        font = self.get_font()._get_q_font()
        #font_color = self._font._get_q_color()
        
        fm = QFontMetrics(font)
        
        style = QStyle.instance()
        mid_color = style.get_mid_color()
        text_color = style.get_text_color()
        
        qp.setFont(font)
        qp.setPen(text_color)
    
        item_info = self._cached_ids_item_info.get(id, None)
        if item_info is not None:
            text = item_info.caption
            if text is None:
                text = ''
            pixmap = item_info.pixmap
        else:
            if id not in self._cached_ids:
                self._cache_item(id)
            
            text = f'{id}...'
            pixmap = None
    
        image_rect = rect.adjusted(0,0,0,-self._cap_spacing - self._cap_h)
        cap_rect = rect.adjusted(0,rect.height() - self._cap_h,0, 0)
    
        qp.fillRect(image_rect, mid_color)
        qp.fillRect(cap_rect, mid_color)
        
        if pixmap is not None:
            q_pixmap = pixmap._get_q_pixmap()
            
            size = q_pixmap.size()
            if size.width() == 0 or size.height() == 0:
                qp.drawText(image_rect,  Qt.AlignmentFlag.AlignCenter, 'Unviewable')
            else:            
                fitted_image_rect = qt.QRect_fit_in(QRect( QPoint(0,0), q_pixmap.size() ), image_rect)
                qp.drawPixmap(fitted_image_rect, q_pixmap)
        else:
            #
            qp.drawText(image_rect,  Qt.AlignmentFlag.AlignCenter, 'Loading...')
        
        elided_text = fm.elidedText(text, Qt.TextElideMode.ElideRight, cap_rect.width())
        
        qp.drawText(cap_rect,  Qt.AlignmentFlag.AlignCenter, elided_text)
        
        # qp.fillRect(image_rect, style_mid_q_color)
        
