import math
from typing import Set, Union

from .. import mx, qt
from .QStyle import QStyle
from .QWidget import QWidget


class QAbstractLiveItemView(QWidget):
           
    def __init__(self):
        super().__init__()
        
        self._mx_selected_items = mx.Property[Set](set()).dispose_with(self)
        
        self._idx_count = -1
        self._item_w = 0
        self._item_h = 0
        self._spacing = 4
        
        self._col_width = -1
        self._row_height = -1
        self._col_count = -1
        self._row_count = -1
        self._table_l_offset = 0
        self._visible_row_count_f = 0
        
        self._view_row = 0.0
        
        self._visible_idxs_start = 0
        self._visible_idxs_count = 0
        
        self._mouse_down_pt = None
        self._selecting_idx_range = None
        self._selecting_un = False
        self._selected_idxs = set()
        
        self._arrow_up_keyseq = qt.QKeySequence( qt.Qt.Key.Key_Up)
        self._arrow_down_keyseq = qt.QKeySequence( qt.Qt.Key.Key_Down)
        
        self._page_down_keyseq = qt.QKeySequence( qt.Qt.Key.Key_PageDown)
        self._page_up_keyseq = qt.QKeySequence( qt.Qt.Key.Key_PageUp)
        
        self._ctrl_a_keyseq = qt.QKeySequence(  qt.QKeyCombination(qt.Qt.KeyboardModifier.ControlModifier, qt.Qt.Key.Key_A) )
        self._arrow_left_keyseq = qt.QKeySequence( qt.Qt.Key.Key_Left)
        self._arrow_right_keyseq = qt.QKeySequence( qt.Qt.Key.Key_Right)
        
        self.__qp = qt.QPainter()
        q_scrollbar = self.__q_scrollbar = qt.QScrollBar()
        q_scrollbar.setSizePolicy(qt.QSizePolicy.Policy.Fixed, qt.QSizePolicy.Policy.Preferred)
        q_scrollbar.valueChanged.connect(self._on_scrollbar_valueChanged)
        
        q_main_widget = self.get_q_widget()
        qt.wrap(q_main_widget, 'keyPressEvent', self._main_widget_keyPressEvent)
        qt.wrap(q_main_widget, 'keyReleaseEvent', self._main_widget_keyReleaseEvent)
        qt.wrap(q_main_widget, 'wheelEvent', self._main_widget_wheelEvent)
        
        q_canvas_widget = self.__q_canvas_widget = qt.QWidget()
        
        qt.wrap(q_canvas_widget, 'mouseMoveEvent', self._widget_mouseMoveEvent)
        qt.wrap(q_canvas_widget, 'mousePressEvent', self._widget_mousePressEvent)
        qt.wrap(q_canvas_widget, 'mouseReleaseEvent', self._widget_mouseReleaseEvent)
        qt.wrap(q_canvas_widget, 'minimumSizeHint', self._widget_minimumSizeHint)
        qt.wrap(q_canvas_widget, 'resizeEvent', self._widget_resizeEvent)
        qt.wrap(q_canvas_widget, 'paintEvent', self._widget_paintEvent)
        
        
        q_widget_l = qt.QHBox()
        q_widget_l.addWidget(q_canvas_widget)
        q_widget_l.addWidget(q_scrollbar)
        q_main_widget.setLayout(q_widget_l)
        
        self.set_item_size(128, 128)
        self.set_idx_count(0)
    
    @property
    def mx_selected_items(self) -> mx.State[set]:
        return self._mx_selected_items
        
    def _on_paint_item(self, id : int, qp : qt.QPainter, rect : qt.QRect):
        """overridable. Paint item content in given rect."""
        raise NotImplementedError()
        
    def set_item_size(self, w, h):
        self._update_geometry(new_item_w=w, new_item_h=h)
        return self
        
    def set_idx_count(self, item_count):
        self._update_geometry(new_item_count=item_count)
        return self
        
    # def get_idx_row(self, idx : int):
    #     return idx // self._col_count
    
    def get_visible(self) -> Set[int]:
        return set(range(self._visible_idxs_start, self._visible_idxs_start+self._visible_idxs_count))
     
    def get_selecting(self) -> Set[int]:
        if self._selecting_idx_range is not None:
            return set( range(min(self._selecting_idx_range), max(self._selecting_idx_range)+1) )
        else:
            return set()
        
    def get_selected(self) -> Set[int]:
        """"""
        selected_set = self._selected_idxs        
        
        if self._selecting_idx_range is not None:
            selecting_set = self.get_selecting()
        
            if self._selecting_un:
                selected_set = selected_set.difference(selecting_set)
            else:
                selected_set = selected_set.union(selecting_set)
            
        return selected_set
    
    def is_selected(self, idx : int) -> bool:
        return idx in self.get_selected()
    
    def is_selected_all(self) -> bool:
        return len(self.get_selected()) == self._idx_count
        
               
    def select(self, idx : Union[int, Set[int]]):
        if not isinstance(idx, set):
            idx = set([idx])
        
        new_selected_idxs = self._selected_idxs.union(idx)
        if len(new_selected_idxs) != len(self._selected_idxs):
            self._selected_idxs = new_selected_idxs
            self._mx_selected_items.set( new_selected_idxs.copy() )
            self.update()
    
    def unselect(self, idx : Union[int, Set[int]] ):
        """"""
        if not isinstance(idx, set):
            idx = set([idx])
        
        new_selected_idxs = self._selected_idxs.difference(idx)
        
        if len(new_selected_idxs) != len(self._selected_idxs):
            self._selected_idxs = new_selected_idxs
            self._mx_selected_items.set( new_selected_idxs.copy() )
                            
        self.update()
    
    def select_all(self):
        """"""
        if len(self._selected_idxs) != self._idx_count:
            self._selected_idxs = set(range(self._idx_count))
            self._mx_selected_items.set(self._selected_idxs.copy)
            self.update()
            
    def unselect_all(self):
        if len(self._selected_idxs) != 0:
            self._selected_idxs = set()
            self._mx_selected_items.set(set())
            self.update()
                    
        
    def select_prev(self, row_up=False, page_up=False):
        """"""
        selected = self.get_selected()
        if len(selected) == 0:
            selected = self.get_visible()
        
        idx = min(selected)
        if row_up:
            idx -= self._col_count
        elif page_up:
            idx -= (self._visible_row_count_f // 2) * self._col_count
        else:
            idx -= 1
            
        idx = max(0, idx)
        self.unselect_all()
        self.select(idx)
        self.view_idx(idx)
        
    def select_next(self, row_down=False, page_down=False):
        """"""
        selected = self.get_selected()
        if len(selected) == 0:
            selected = self.get_visible()
            
        idx = max(selected)
        if row_down:
            idx += self._col_count
        elif page_down:
            idx += (self._visible_row_count_f // 2) * self._col_count
        else:
            idx += 1
            
        idx = min(idx, self._idx_count-1)
        self.unselect_all()
        self.select(idx)
        self.view_idx(idx)
        
    
    def view_row(self, row : int):
        self._update_geometry(new_view_row=row)
        
    def view_idx(self, id : int):
        self.view_row(id // self._col_count)
        
    def _on_scrollbar_valueChanged(self, value):
        self._update_geometry( new_view_row=value )
        
    def _get_idx_from_point(self, point : qt.QPoint) -> Union[int, None]:
        view_x = point.x()
        view_y = point.y()
        spacing = self._spacing       
        
        item_x = view_x - spacing//2 - self._table_l_offset
        item_y = view_y - (self._visible_row_count_f/2.0-self._view_row)*self._row_height + self._item_h/2.0

        idx = int( math.floor(item_y / self._row_height) * self._col_count + item_x // self._col_width )

        if idx >= 0 and idx < self._idx_count:
            return idx
        return None
    
    def _get_item_rect(self, idx) -> qt.QRect:
        spacing = self._spacing        
            
        item_x = (idx  % self._col_count)*self._col_width  
        item_y = (idx // self._col_count)*self._row_height 
        
        view_x = item_x + spacing//2 + self._table_l_offset
        view_y = item_y + (self._visible_row_count_f/2.0-self._view_row)*self._row_height - self._item_h/2.0
        
        return qt.QRect(qt.QPoint(view_x,view_y), qt.QSize(self._item_w, self._item_h))
        
    
    def _update_geometry(self,  new_item_count = None,
                                new_item_w = None,
                                new_item_h = None,
                                new_spacing = None,
                                new_view_row = None):
        rect = self.__q_canvas_widget.rect()
        
        w = rect.width()
        h = rect.height()
        
        item_count  = self._idx_count
        new_item_count = self._idx_count = new_item_count if new_item_count is not None else item_count
        
        item_w     = self._item_w
        new_item_w = self._item_w = new_item_w if new_item_w is not None else item_w
        
        item_h     = self._item_h
        new_item_h = self._item_h = new_item_h if new_item_h is not None else item_h
        
        spacing     = self._spacing
        new_spacing = self._spacing = new_spacing if new_spacing is not None else spacing
        
        col_width     = self._col_width
        new_col_width = self._col_width = new_item_w + new_spacing
        
        row_height     = self._row_height
        new_row_height = self._row_height = new_item_h + new_spacing
        
        col_count     = self._col_count 
        new_col_count = self._col_count = max(1, w // new_col_width)
        
        self._table_l_offset = (w - new_col_count*new_col_width) // 2
        
        row_count     = self._row_count
        new_row_count = self._row_count = math.ceil(new_item_count / new_col_count)
        
        view_row     = self._view_row
        new_view_row = self._view_row = max(0, min(new_view_row if new_view_row is not None else view_row, new_row_count-1) )
        
        new_visible_row_count_f = self._visible_row_count_f = h / new_row_height
        
        visible_idxs_start     = self._visible_idxs_start 
        new_visible_idxs_start = self._visible_idxs_start = max(0, min(new_item_count-1,
                                                                int(math.ceil(new_view_row - new_visible_row_count_f/2.0))*new_col_count))
        
        visible_idxs_count     = self._visible_idxs_count #
        new_visible_idxs_count = self._visible_idxs_count = max(0, min(new_item_count,                                                                    
                                                                max(0, min(new_item_count, 
                                                                    int(math.ceil(new_view_row + new_visible_row_count_f/2.0))*new_col_count))
                                                                - new_visible_idxs_start))
        # print('new_view_row', new_view_row)
        # print('new_item_count', new_item_count)      
        # print('new_visible_row_count_f', new_visible_row_count_f)      
        # print('new_visible_idxs_start', new_visible_idxs_start)      
        # print('new_visible_idxs_count', new_visible_idxs_count)      
        
        if new_row_count != row_count:
            if new_row_count == 0:
                self.__q_scrollbar.hide()
            else:                
                self.__q_scrollbar.show()
                self.__q_scrollbar.setMinimum(0)
                self.__q_scrollbar.setMaximum( new_row_count-1 )
            
        if new_view_row != view_row:
            self.__q_scrollbar.setValue(int(new_view_row))
           
        if new_col_count != col_count:
            prev_center_item = round( col_count * ( view_row + 0.5) )
            self.view_idx(prev_center_item)
            
        if new_item_w - item_w + new_item_h - item_h + new_item_count - item_count != 0:
            self.update_geometry()
            
        self.update()
            
    def _widget_minimumSizeHint(self, widget, super ) -> qt.QSize:
        return qt.QSize(self._item_w+self._spacing, self._item_h+self._spacing)
                        
    def _widget_resizeEvent(self, widget, super, ev : qt.QResizeEvent):
        super(ev)
        self._update_geometry()
    
    def _widget_mousePressEvent(self, widget, super, ev : qt.QMouseEvent):
        super(ev)
        mouse_pt = ev.pos()
        self._mouse_down_pt = mouse_pt
        
        keymods = qt.QApplication.keyboardModifiers()
        ctrl_pressed = keymods == qt.Qt.KeyboardModifier.ControlModifier
        
        idx = self._get_idx_from_point(mouse_pt)
        
        if idx is not None:
            self._selecting_un = self.is_selected(idx)
            self._selecting_idx_range = (idx,idx)
        else:
            self._selecting_idx_range = None
            
        if not ctrl_pressed:
            self.unselect_all()
            
        self.update()
            
    
    def _widget_mouseMoveEvent(self, widget, super, ev : qt.QMouseEvent):
        super(ev)
        mouse_pt = ev.pos()
        
        item_id_pt = self._get_idx_from_point(mouse_pt)
        
        if self._selecting_idx_range is not None and item_id_pt is not None:
            self._selecting_idx_range = (self._selecting_idx_range[0], item_id_pt)
            self.update()
        
        #print( QLineF(mouse_pt, self._mouse_down_pt).length() )
        
        
    def _widget_mouseReleaseEvent(self, widget, super, ev : qt.QMouseEvent):
        super(ev)
        mouse_pt = ev.pos()
        self._mouse_down_pt = None
        
        item_id_pt = self._get_idx_from_point(mouse_pt)
        
        if self._selecting_idx_range is not None and item_id_pt is not None:
            self._selecting_idx_range = (self._selecting_idx_range[0], item_id_pt)
            self.update()
          
        if self._selecting_idx_range is not None:
            selected_set = self.get_selecting()
            
            if self._selecting_un:
                self.unselect(selected_set)
            else:                
                self.select(selected_set)
                
            self._selecting_idx_range = None
             
            
    def _main_widget_wheelEvent(self, widget, super, ev : qt.QWheelEvent):
        d = ev.angleDelta().y()
        
        self._update_geometry( new_view_row=self._view_row + (1.0 if d < 0 else -1.0) )
        self.update()
    
    def _main_widget_keyPressEvent(self, widget, super, ev : qt.QKeyEvent):
        #
        key_combination = ev.keyCombination()
        
        if self._page_down_keyseq.matches(key_combination):
            self.select_next(page_down=True)
        elif self._page_up_keyseq.matches(key_combination):
            self.select_prev(page_up=True)
        elif self._ctrl_a_keyseq.matches(key_combination):
            if self.is_selected_all():
                self.unselect_all()
            else:
                self.select_all()
        elif self._arrow_left_keyseq.matches(key_combination):
            self.select_prev()
        elif self._arrow_right_keyseq.matches(key_combination):
            self.select_next()
        elif self._arrow_up_keyseq.matches(key_combination):
            self.select_prev(row_up=True)
        elif self._arrow_down_keyseq.matches(key_combination):
            self.select_next(row_down=True)
        else:
            super(ev)
        
    def _main_widget_keyReleaseEvent(self, widget, super, ev : qt.QKeyEvent):
        key_combination = ev.keyCombination()
        if (self._page_down_keyseq.matches(key_combination) or
            self._page_up_keyseq.matches(key_combination) or
            self._ctrl_a_keyseq.matches(key_combination) or
            self._arrow_left_keyseq.matches(key_combination) or 
            self._arrow_right_keyseq.matches(key_combination) or
            self._arrow_up_keyseq.matches(key_combination) or 
            self._arrow_down_keyseq.matches(key_combination)
            ):
            ...
        else:
            super(ev)
            
    def _widget_paintEvent(self, widget, super, ev : qt.QPaintEvent):
            
        q_widget = self.__q_canvas_widget
        
        qp = self.__qp
        qp.begin(q_widget)
        
        #
        
        # w = rect.width()
        # h = rect.height()
        
        # cap_h = self._cap_h
        # item_w = self._item_w
        # item_h = self._item_h
        # space = self._spacing
        # scroll_y = self._scroll_y
        # row_items_max = self._row_count
        
        style = QStyle.instance()
        selection_frame_q_color = style.get_selected_color()
        
        if self._visible_idxs_count != 0:
            for idx in range(self._visible_idxs_start, self._visible_idxs_start+self._visible_idxs_count):
                
                item_rect = self._get_item_rect(idx)
                
                if self.is_selected(idx):
                    item_outter_rect = item_rect.marginsAdded( qt.QMargins(1,1,1,1))
                    qp.fillRect(item_outter_rect, selection_frame_q_color)
                    
                    # pen_item_selected = QPen( Qt.GlobalColor.white )    
                    # pen_item_selected.setWidth(self._spacing)
                    # qp.setPen(pen_item_selected)
                    # qp.drawRect(item_outter_rect)
                
                self._on_paint_item(idx, qp, item_rect)
        else:
            rect = q_widget.rect()
            
            qp.drawText(rect, qt.Qt.AlignmentFlag.AlignCenter, 'no items')    
            
        qp.end()