from numbers import Number
from pathlib import Path
from typing import Any, Callable, Union

import cv2
import numpy as np

from ... import qx
from xlib import mx
from xlib import ax
from ...Box import Box
from ...xlib import abstract


class VideoPlayerBox(Box):
    
    def __init__(self, relative_path : Path = None, 
                       on_play_start : Callable[ [Any], ax.Task ] = None,
                       on_play_stop : Callable[ [Any], ax.Task ] = None,
                       on_frame : Callable[ [int, int, np.ndarray], ax.Task ] = None,
                       state : dict = None):
        """
        
            on_frame(frame_idx, frame_count, frame : np.ndarray)       
                                taskmethod which will be initiated when new frame is ready.
                                It always waits while task is finished before push new frame.
                                If con_play_mode is MediaSourceBox.PlayMode.realtime,
                                then it emits current frame in timeline. 
                                If con_play_mode is MediaSourceBox.PlayMode.every_frame,
                                then it emits next frame in timeline.
                                
        """
        super().__init__()
        self._relative_path = relative_path
        self._on_frame = on_frame
        
        self._cap = None
        self._cap_tr = ax.TaskGroup()
        
        self._fp : abstract.FramePlayer = None
        self._ev_upd_fp_section = mx.Event0()
  
        self._con_error_stream = mx.TextStream()
        
        self._con_seek_begin_signal = None
        self._con_seek_backward_signal = None
        self._con_play_signal = None
        self._con_stop_signal = None
        self._con_seek_forward_signal = None
        self._con_seek_end_signal = None
        
        
        self._con_frame_index = None
        self._con_frame_count = None
        
        self._con_media_path = mx.Path(is_directory=False, relative_path=self._relative_path, control_func=self._media_path_control)
        self._con_media_path.set_state( (state or {}).get('_con_media_path', None))
        
        
    def finalize(self):  
        super().finalize()
        self._con_media_path.set_path(None)
        self._con_media_path.finalize()
        self._con_error_stream.finalize()
        
        
    def _media_path_control(self, new_path : Union[Path, None], path : Union[Path, None]):
        print('_media_path_control', new_path)
        self._cap_tr.cancel_all()
        
        self._fp = None
        if self._cap is not None:
            self._cap.release()
            self._cap = None
         
        self._con_seek_begin_signal = self._con_seek_begin_signal.finalize() if self._con_seek_begin_signal is not None else None
        self._con_seek_backward_signal = self._con_seek_backward_signal.finalize() if self._con_seek_backward_signal is not None else None   
        self._con_play_signal = self._con_play_signal.finalize() if self._con_play_signal is not None else None
        self._con_stop_signal = self._con_stop_signal.finalize() if self._con_stop_signal is not None else None
        self._con_seek_forward_signal = self._con_seek_forward_signal.finalize() if self._con_seek_forward_signal is not None else None
        self._con_seek_end_signal = self._con_seek_end_signal.finalize() if self._con_seek_end_signal is not None else None   
                
        self._con_frame_index = self._con_frame_index.finalize() if self._con_frame_index is not None else None
        self._con_frame_count = self._con_frame_count.finalize() if self._con_frame_count is not None else None
        
        if new_path is not None:
            
            cap = cv2.VideoCapture()
            cap.open(str(new_path), cv2.CAP_FFMPEG)          
            if cap.isOpened():      
                self._cap = cap
                fp = self._fp = abstract.FramePlayer(   frame_idx=int(cap.get(cv2.CAP_PROP_POS_FRAMES )), 
                                                        frame_count=int(cap.get(cv2.CAP_PROP_FRAME_COUNT)), 
                                                        fps=cap.get(cv2.CAP_PROP_FPS) )
                
                self._con_seek_begin_signal = mx.SignalEmitter()
                self._con_seek_begin_signal.on_signal.subscribe(self._on_con_seek_begin_signal)
                
                self._con_seek_backward_signal = mx.SignalEmitter()
                self._con_seek_backward_signal.on_signal.subscribe(self._on_con_seek_backward_signal)
                
                self._con_play_signal = mx.SignalEmitter()
                self._con_play_signal.on_signal.subscribe(self._on_con_play_signal)
                
                self._con_stop_signal = mx.SignalEmitter()
                self._con_stop_signal.on_signal.subscribe(self._on_con_stop_signal)
                
                self._con_seek_forward_signal = mx.SignalEmitter()
                self._con_seek_forward_signal.on_signal.subscribe(self._on_con_seek_forward_signal)
                
                self._con_seek_end_signal = mx.SignalEmitter()
                self._con_seek_end_signal.on_signal.subscribe(self._on_con_seek_end_signal)
                
                self._con_frame_index = mx.Number(range=(0,fp.frame_count-1), control_func=self._on_con_frame_index)
                self._con_frame_index.set_number(fp.frame_idx, bypass_control=True)
                
                self._con_frame_count = mx.Number(range=(fp.frame_count-1, fp.frame_count-1))
                self._con_frame_count.set_number(fp.frame_count-1, bypass_control=True)
                
                self._cap_task()
            else:
                self._con_error_stream.push(f'Unable to open {new_path}')
                return None
            
        self._rebuild_gui()            
        return new_path
        
    @ax.task
    def _cap_task(self):
        yield ax.attach_to(self._cap_tr)
        
        cap_thread = ax.Thread('cap_thread')
        
        try:
            yield ax.switch_to(self.get_main_thread())
            
            while True:
                r = self._fp.process()
                if r.new_is_playing is not None:
                    if r.new_is_playing:
                        ...
                        #self._gui_play_btn.enable()
                    else:
                        ...
                    self._ev_upd_fp_section.async_emit()
                    
                new_frame = r.new_frame
                if new_frame is not None:
                    frame_idx = new_frame.frame_idx
                    
                    """
                    #init
                    self._con_frame_index_req = mx.Flag(False)
                    
                    self._con_frame_index = mx.Number(0, sign=self
                    self._con_frame_index.listen(lambda n,o: self._con_frame_index_req.set(True))
                    
                    ##
                    if self._con_frame_index_req.value:
                        self._con_frame_index_req.set(False)
                        new_frame_idx=self._con_frame_index.value
                    
                    self._con_frame_index.set(frame_idx)
                    self._con_frame_index_req.set(False)
                    
                    
                    ##
                    
                    self._con_frame_index = mx.Number(0, sign=self)
                    
                    ###
                    
                    if not (self._con_frame_index.sign is self):
                        new_frame_idx=self._con_frame_index.value
                    
                    self._con_frame_index.set(frame_idx, sign=self)
                    
                    #
                    
                    
                    """
                    
                    self._con_frame_index.set_number(frame_idx, bypass_control=True)#sign=1
                    
                    self._con_error_stream.push(f'Frame {frame_idx}/{self._fp.frame_count}')
                    
                    yield ax.switch_to(cap_thread)
                    
                    if not self._cap.set(cv2.CAP_PROP_POS_FRAMES, frame_idx):
                        #self. run_in_thread ( lambda: )
                        self._con_error_stream.async_emit(f'Error while seeking to {frame_idx} frame.')
                        self._con_media_path.async_set_path(None)
                        break
                        
                    ret, frame = self._cap.read()
                    if not ret:
                        self._con_error_stream.async_emit(f'Error while reading {frame_idx} frame.')
                        self._con_media_path.async_set_path(None) 
                        break
                        
                    yield ax.switch_to(self.get_main_thread())
                    
                    if self._on_frame is not None:
                        yield ax.wait( self._on_frame(frame_idx, self._fp.frame_count, frame) )
                        
                    
                
                yield ax.sleep(0)
                
        except ax.TaskFinishError:
            ...
            
        cap_thread.finalize()
    
    
    def _on_con_seek_begin_signal(self):
        self._fp.req_frame_seek_idx = (0,0)
    
    def _on_con_seek_backward_signal(self):
        self._fp.req_frame_seek_idx = (-1,1)
        
    def _on_con_play_signal(self):
        self._fp.req_is_playing = True
        self._con_error_stream.push('Play start')
            
    def _on_con_stop_signal(self):
        self._fp.req_is_playing = False
        
    def _on_con_seek_forward_signal(self):
        self._fp.req_frame_seek_idx = (1,1)
         
    def _on_con_seek_end_signal(self):
        self._fp.req_frame_seek_idx = (0,2)
            
    def _on_con_frame_index(self, new_index : Union[Number, None], index : Union[Number, None] ):
        self._fp.req_frame_seek_idx = (new_index, 0)
    
    @ax.task
    def get_state(self) -> ax.Task[dict]:
        return {'_con_media_path' : self._con_media_path.get_state()}
        
    def _fp_section(self, gui : qx.XBuilder):
        fp = self._fp
        if fp is not None:
        
            with gui.hstack() as hst, gui.size(qx.ESize.S3), gui.color(qx.EColor.red):
                gui.spacer().expand(h=True)
                
                gui.pushbutton(self._con_seek_begin_signal, image=gui.get_ionicon_image(qx.EIonIcon.play_skip_back_circle_outline))
                gui.pushbutton(self._con_seek_backward_signal, image=gui.get_ionicon_image(qx.EIonIcon.play_back_circle_outline))
                
                play_btn = gui.pushbutton(self._con_play_signal, image=gui.get_ionicon_image(qx.EIonIcon.play_circle_outline), tooltip='Play')
                stop_btn = gui.pushbutton(self._con_stop_signal, image=gui.get_ionicon_image(qx.EIonIcon.pause_circle_outline), tooltip='Stop' )
                if fp.is_playing:
                    play_btn.disable()
                else:
                    stop_btn.disable()
                    
                gui.pushbutton(self._con_seek_forward_signal, image=gui.get_ionicon_image(qx.EIonIcon.play_forward_circle_outline))
                gui.pushbutton(self._con_seek_end_signal, image=gui.get_ionicon_image(qx.EIonIcon.play_skip_forward_circle_outline))
                    
                gui.spacer().expand(h=True)
                
            with gui.hstack() as hst:
                gui.spacer()
                with gui.hstack() as hst:
                    hst.compact()
                    gui.spinbox(self._con_frame_index)
                    gui.spinbox(self._con_frame_count, read_only=True)
                gui.spacer()
                
    def _on_gui(self, gui : qx.XBuilder): 
        with gui.vstack() as vst:
            vst.compact(v=True)
            gui.label('Video Player')
            gui.path(self._con_media_path) 
            gui.section(self._fp_section, self._ev_upd_fp_section)
            
            with gui.hstack() as hst:
                with gui.font(font=qx.EFontDB.FixedWidth):                
                    gui.message_stream(self._con_error_stream, image=gui.get_ionicon_image(qx.EIonIcon.warning_outline, qx.EColor.red, qx.ESize.S2) )
                      
                      
                      
                            
            # with gui.vstack():
                
            #     with gui.hstack():
            #         gui.label('Source type').compact(h=True)
            #         gui.combobox(self._con_source_type)
                
            #     source_type = self._con_source_type.get_choice()
            #     if source_type == SourceType.IMAGE_SEQUENCE:
            #         gui.path(self._con_media_img_seq_path)
            #     elif source_type == SourceType.VIDEO_FILE:
            #         gui.path(self._con_media_video_path)   
                    
            #     fp = self._fp
            #     if fp is not None:
            #         if not fp.is_initialized():
            #             gui.label('Initializing...')
            #         else:
            #             ...
            """
            fp_is_playing = fp.is_playing()
            
            gui.button(icon=gui.icons.play, disabled=fp_is_playing)
            """
                

# class SourceType(IntEnum):
#     IMAGE_SEQUENCE = 0
#     VIDEO_FILE = 1
#     CAMERA = 2
    
#     def __str__(self, lang : str = 'en'):
#         return SourceType_name.get(lang if lang in SourceType_name.keys() else 'en')[self]
        
# SourceType_name = {'en':{ SourceType.IMAGE_SEQUENCE : 'Image Sequence',
#                           SourceType.VIDEO_FILE : 'Video File',
#                           SourceType.CAMERA : 'Camera',
#                         }
#                   }

#ax.lib.FramePlayer()

# class ImageSequencePlayer:
#     def __init__(self, sequence_path : Path, on_initialized : Callable):
#         self._sequence_path = sequence_path
#         self._on_initialized = on_initialized
        
#         self._ev_play_start = mx.Event0()
#         self._ev_play_stop = mx.Event0()
#         self._ev_frame = mx.Event1()
        
#         self._ts = ax.Taskset()
#         self._main_thread = ax.get_current_thread()
#         self._fp_thread = ax.Thread('FramePlayer_fp_thread')
        
#         self._initialized = False
#         self._initialization_task()
        
#     def is_initialized(self): return self._initialized
        
#     @property
#     def ev_play_start(self): return self._ev_play_start
    
#     @property
#     def ev_play_stop(self): return self._ev_play_stop
    
#     @property
#     def ev_frame(self): return self._ev_frame
    
#     def finalize(self):
#         self._ts.finalize()
#         self._fp_thread.finalize()
        
#     @ax.task
#     def _initialization_task(self) -> ax.Task:
#         yield ax.attach_to(self._ts)
#         yield ax.switch_to(self._fp_thread)
        
#         self._filespaths = ext_path.get_files_paths(self._sequence_path, extensions=['.jpeg','.jpg','.png'])
#         self._fp = abstract.FramePlayer(frame_count=len(self._filespaths))
        
#         yield ax.switch_to(self._main_thread)
        
#         self._initialized = True
#         self._on_initialized()
#         self._process_task()
        
#     def get_frame_count(self) -> int: return self._fp.get_frame_count()
    
#     def req_play_start(self): self._fp.req_play_start()
#     def req_play_stop(self): self._fp.req_play_stop()
#     def req_frame_seek(self, idx : int, mode : int): self._fp.req_frame_seek(idx, mode)
                
#     @ax.task
#     def _process_task(self) -> ax.Task:
#         yield ax.attach_to(self._ts)
        
#         yield ax.switch_to(self._fp_thread)
#         while True:
#             pr = self._fp._process()
#             if pr.play_start:
#                 yield ax.switch_to(self._main_thread)
#                 self.ev_play_start.push()
                
#             if pr.play_stop:
#                 yield ax.switch_to(self._main_thread)
#                 self.ev_play_stop.push()
            
#             # if pr.new_frame is not None:
#             #     pr.new_frame
            
#             yield ax.switch_to(self._fp_thread)
            
#             yield ax.sleep(0.016)
            
                
    
    
    # @ax.task
    # def _process_task(self) -> ax.Task:
    #     yield ax.attach_to(self._ts)
        
    #     yield ax.switch_to(self._fp_thread)
        
    #     fl = self._fl
    #     while True:
            
    #         pr = fl.process()
            
            
    #         yield ax.sleep(0.016)
    
        #self._media_path_recent_img_seq = state.get('media_path_recent_img_seq', [])
        #self._media_path_recent_video = state.get('media_path_recent_video', [])
        
        #self._con_media_video_path   = mx.Path(is_directory=False, relative_path=self._root_path, control_func=self._media_video_path_control)
                                #mx.Path(is_directory=True,  relative_path=self._relative_path, control_func=self._media_img_seq_path_control)
        #self._con_source_type = mx.SingleChoice(avail_choices=SourceType, control_func=lambda *args, state=state: self._con_source_type_control(*args, state=state))
        #self._con_source_type.set_choice( SourceType(state.get('source_type', 0)))
        