import math
import pickle
import shutil
import time
from numbers import Number
from pathlib import Path
from typing import Callable, Union

from xlib import mx
from xlib import ax
from ...xlib import path as ext_path
from ...Box import Box
from ... import qx


class FileStateBox(Box):
    
    def __init__(self, state_path : Path,
                       task_on_load_state : Callable[ [ dict ], ax.Task ],
                       task_on_get_state  : Callable[ [], ax.Task[dict] ], ):
        """Provides loadable/saveable state of picklable dict"""
        super().__init__()
        self._tr = ax.TaskGroup()
        self._save_tr = ax.TaskGroup()
        self._io_thread = ax.Thread(name='IO')
        
        self._task_on_load_state = task_on_load_state
        self._task_on_get_state = task_on_get_state
        
        self._con_progress = None
        self._con_autosave_every_mins = None
        self._con_autobackup_every_mins = None
        self._con_backups_max_count = None 
        self._con_load_sig = None
        self._con_save_sig = None
        self._con_backup_sig = None
        self._con_load_menu = None
        self._reload(state_path)
        
    def finalize(self):
        super().finalize()
        self._finalize_controls()
        
        self._save_tr.finalize()
        self._tr.finalize()
        self._io_thread.finalize()
    
        
    
    
    @ax.task
    def _reload(self, state_path : Path) -> ax.Task:
        self._tr.cancel_all()
        yield ax.attach_to(self._tr)
        
        self._finalize_controls()
        self._loaded = False

        yield ax.switch_to(self.get_main_thread())
        
        name_split = state_path.name.split('.')
        self._state_path      = state_path.parent / f'{name_split[0]}.{name_split[-1]}'
        self._state_path_stem = self._state_path.stem
        self._state_path_part = self._state_path.parent / (self._state_path.name + '.part')
        self._state_root_path = self._state_path.parent
        
        self._con_progress = mx.UniformProgress()
        self._con_progress.start()
        self._con_progress.progress(0.5)
        self._rebuild_gui()
        
        yield ax.switch_to(self._io_thread)
        
        state = {}
        if state_path.exists():
            with open(state_path, 'rb') as file:
                state = pickle.load(file)
                
        yield ax.switch_to(self.get_main_thread())
        
        self._con_autosave_every_mins = mx.Number(range=(0,3600), step=1, control_func=self._autosave_every_mins_control)
        self._con_autosave_every_mins.set_number( state.get('autosave_every_mins', 25) )
        
        self._con_autobackup_every_mins = mx.Number(range=(0,3600), step=1, control_func=self._autobackup_every_mins_control)
        self._con_autobackup_every_mins.set_number( state.get('autobackup_every_mins', 0) )

        self._con_backups_max_count = mx.Number(range=(1,32), step=1)
        self._con_backups_max_count.set_number( state.get('backups_max_count', 8) )
                        
        self._con_load_sig = mx.SignalEmitter()
        
        con_save_sig = self._con_save_sig = mx.SignalEmitter()
        con_save_sig.on_signal.subscribe(lambda: self._save())
        
        con_backup_sig = self._con_backup_sig = mx.SignalEmitter()
        con_backup_sig.on_signal.subscribe(lambda: self._save(backup=True))
        
        load_menu = self._con_load_menu = mx.Menu(avail_choices=lambda: [ StatePath(filepath) for filepath in ext_path.get_files_paths(self._state_root_path) if (f'{self._state_path_stem}.' in filepath.name) and (filepath.suffix == '.state')])
        load_menu.on_choice.subscribe(lambda choice, avail_choices: self._reload(choice.path) )
        
        yield ax.wait( self._task_on_load_state(state.get('user_state', {})) )

        self._con_progress.finish()
        self._loaded = True
        self._rebuild_gui()
        self._auto_save_task() 
        
    def _finalize_controls(self):
        self._con_autosave_every_mins = self._con_autosave_every_mins.finalize() if self._con_autosave_every_mins is not None else None
        self._con_autobackup_every_mins = self._con_autobackup_every_mins.finalize() if self._con_autobackup_every_mins is not None else None
        self._con_backups_max_count = self._con_backups_max_count.finalize() if self._con_backups_max_count is not None else None
        self._con_load_sig = self._con_load_sig.finalize() if self._con_load_sig is not None else None
        self._con_save_sig = self._con_save_sig.finalize() if self._con_save_sig is not None else None
        self._con_backup_sig = self._con_backup_sig.finalize() if self._con_backup_sig is not None else None
        self._con_load_menu = self._con_load_menu.finalize() if self._con_load_menu is not None else None
        self._con_progress = self._con_progress.finalize() if self._con_progress is not None else None
        
                
    def _autosave_every_mins_control(self, new_mins : Union[Number, None], mins : Union[Number, None]):
        if mins in [0, None]:
            self._last_save_time = time.time()
        return new_mins
        
    def _autobackup_every_mins_control(self, new_mins : Union[Number, None], mins : Union[Number, None]):
        if mins in [0, None]:
            self._last_backup_time = time.time()
        return new_mins
           
    @ax.task
    def _auto_save_task(self) -> ax.Task:  
        yield ax.attach_to(self._tr)
        yield ax.switch_to(self.get_main_thread())
        
        while True:
            autosave_every_mins = self._con_autosave_every_mins.get_number()
            if autosave_every_mins != 0:
                if ( time.time() - self._last_save_time ) >= autosave_every_mins*60 :
                    yield ax.wait(self._save())
            
            autobackup_every_mins = self._con_autobackup_every_mins.get_number()
            if autobackup_every_mins != 0:
                if ( time.time() - self._last_backup_time ) >= autobackup_every_mins*60 :
                    yield ax.wait(self._save(backup=True))
                    
            yield ax.sleep(1)
            
    @ax.task
    def _save(self, backup=False) -> ax.Task:
        self._save_tr.cancel_all()
        
        yield ax.attach_to(self._tr)
        yield ax.attach_to(self._save_tr)
        yield ax.switch_to(self.get_main_thread())
                
        self._last_save_time = time.time()
        if backup:
            self._last_backup_time = self._last_save_time
        
        try:
            file = None
            self._con_progress.start()
                
            yield ax.sleep(0)
            
            user_state = self._task_on_get_state()                
            yield ax.wait(user_state)
            if user_state.succeeded:
                user_state = user_state.result
            else:
                yield ax.cancel()
                   
            state = {}
            state['user_state'] = user_state
            state['autosave_every_mins'] = self._con_autosave_every_mins.get_number()
            state['autobackup_every_mins'] = self._con_autobackup_every_mins.get_number()
            state['backups_max_count'] = self._con_backups_max_count.get_number()
                  
            yield ax.switch_to(self._io_thread)
            
            mv = memoryview(pickle.dumps(state))
            mv_size = len(mv)
            chunks_count = 50
            mv_chunk_size = math.ceil(mv_size / chunks_count)

            file = open(self._state_path_part, 'wb')

            for i in range(chunks_count):
                chunk_end = min( (i+1)*mv_chunk_size, mv_size )
                file.write( mv[i*mv_chunk_size: chunk_end])

                yield ax.switch_to(self.get_main_thread())

                self._con_progress.progress( i / (chunks_count-1) )

                yield ax.switch_to(self._io_thread)
            
            file.close()
            file = None

            yield ax.switch_to(self.get_main_thread())
            
            #TODO os sync save
            if self._state_path.exists():
                self._state_path.unlink()
            self._state_path_part.rename(self._state_path) 
            
            if backup:                       
        
                yield ax.switch_to(self._io_thread)
                
                backups_max_count = self._con_backups_max_count.get_number()
                
                stem, suffix = self._state_path.stem, self._state_path.suffix[1:]
                
                # Delete redundant backups
                for filepath in ext_path.get_files_paths(self._state_root_path):
                    splits = filepath.name.split(f'{stem}.bckp_')
                    if len(splits) == 2:
                        splits = splits[1].split('.')
                        if len(splits) == 2 and splits[1] == suffix:
                            backup_id = int(splits[0])
                            if backup_id > backups_max_count:
                                filepath.unlink()
                
                # Renaming existing backups to free backup slot 01                 
                for i in range(backups_max_count-1,-1,-1):
                    p1 = self._state_root_path / f'{stem}.bckp_{i:02}.{suffix}'
                    p2 = self._state_root_path / f'{stem}.bckp_{i+1:02}.{suffix}'
                    if p2.exists():
                        p2.unlink()
                    if p1.exists():
                        p1.rename(p2)
                
                # Copy saved state file to backup slot 01
                bckp_filename = f'{stem}.bckp_01.{suffix}'
                shutil.copy(self._state_path, self._state_root_path / bckp_filename)
                
                yield ax.switch_to(self.get_main_thread())
                
            self._con_progress.finish()    
            
            yield ax.sleep(0)  
            
        except ax.TaskFinishError:
            if file is not None:
                file.close()
                file = None
            if self._state_path_part.exists():
                self._state_path_part.unlink()
        
    
    def _rr_gui(self): #, parent : qx.Object
        root = self._get_rootwidget()
        
        vst = qx.VStack([ ])
        
        grid = qx.Grid()
        
        load_btn = self._load_btn = qx.PushButton(self._con_load_sig, text='Load')
        grid.add(load_btn)
        #^ grid._childs_by_loc[ (r,c,rs,cs)] = child
        # child.set_parent(self)
        # grid.ev_add_child
        # handler: child is Widget, grid._get_loc_by_widget()
        # child.ev_finalize
        # q_widget.setParent(None)      
        grid.next_row()
        
        #self._fp_section = qx.Widget() 
        
        #self._get_gui_root()
        
        #_rebuild_fp_section
        #self._fp_section.delete_childs()
        #fpsection.add ( VStack )
        #self._fp_play = None
        #
        
        #_fp_section_on_gui( section : qx.Layout)
        # 
        
        
    def _on_gui(self, gui : qx.XBuilder):

        
        with gui.vstack():
            gui.spacer()
            with gui.hstack() as hst: 
                hst.compact(v=True)
                
                gui.spacer()
                with gui.grid() as grid:
                    grid.compact()
                    if self._loaded:
                        load_btn = gui.pushbutton(self._con_load_sig, text='Load')
                        with grid.span(col=2):
                            gui.path(self._state_path)
                            
                        grid.next_row()                    
                        gui.pushbutton(self._con_save_sig, text='Save') 
                        with grid.span(col=2), gui.hstack() as hst:
                            gui.label(f'Every')
                            gui.spinbox(self._con_autosave_every_mins)
                            gui.label(f'minutes')          
                        gui.spacer()
                        
                        grid.next_row()                    
                        gui.pushbutton(self._con_backup_sig, text='Backup' ) 
                    
                        with grid.span(col=2), gui.hstack() as hst:
                            gui.label(f'Every')
                            gui.spinbox(self._con_autobackup_every_mins)
                            gui.label(f'minutes')
                            gui.label('Max')
                            gui.spinbox(self._con_backups_max_count)
                            gui.label('backups')
                    else:
                        with grid.span(col=3):
                            gui.label('Loading')
                    
                    grid.next_row()
                        
                    if self._con_progress is not None:
                        with grid.span(col=3):
                            gui.progressbar(self._con_progress)
                
                gui.spacer()
            
            gui.spacer()
        
    
        if self._loaded:
            with gui.font(qx.EFontDB.FixedWidth):
                gui.popup_menu(self._con_load_sig.on_signal, self._con_load_menu, parent_object=load_btn)
        
        

class StatePath:
    def __init__(self, path):
        self.path = path
        
    def __eq__(self, o):
        return self.path == o.path
        
    def __str__(self):
        return f"[{ext_path.creation_date(self.path).strftime('%Y.%m.%d %H:%M:%S')}] [{ self.path.stat().st_size // (1024**2) }Mb] {self.path.name} "
    