from typing import Any, Callable, Dict, List, Union

from typeguard import typechecked

from ..device import TorchDeviceInfo, get_cpu_device_info
from ..optim import AdaBelief
from .module import init_and_load


class MultiModule:

    @typechecked
    def __init__(self, init_func_by_submodule_type : Dict[Any, Callable],
                       state : Union[dict, None] = None,):
        self._train_or_eval = ...
        self._device_info = get_cpu_device_info()

        self._sub_modules = {}
        self._sub_modules_opt = {}
        self._init_func_by_submodule_type = init_func_by_submodule_type
        self._sub_module_types = init_func_by_submodule_type.keys()

        for submodule_type in self.get_submodule_types():
            self.reset_submodule(submodule_type, state)

    def get_submodule(self, submodule_type : Any): return self._sub_modules[submodule_type]
    def get_submodule_opt(self, submodule_type : Any) -> AdaBelief: return self._sub_modules_opt[submodule_type]
    def get_submodule_types(self) -> List[Any]: return self._sub_module_types

    def state_dict(self) -> dict:
        d = {}
        for submodule_type in self.get_submodule_types():
            d[f'{submodule_type}_state'] = self._sub_modules[submodule_type].state_dict()
            d[f'{submodule_type}_opt_state']= self._sub_modules_opt[submodule_type].state_dict()
        return d

    @typechecked
    def reset_submodule(self, submodule_type : Any, state : dict = None):
        if submodule_type not in self.get_submodule_types():
            raise ValueError(f'submodule_type {submodule_type} not in avail {self.get_submodule_types()}')

        module = self._sub_modules[submodule_type] = init_and_load(self._init_func_by_submodule_type[submodule_type], (state or {}).get(f'{submodule_type}_state', None) )
        module.to(self._device_info.get_device())
        module_opt = self._sub_modules_opt[submodule_type] = init_and_load(lambda: AdaBelief(module.parameters()), (state or {}).get(f'{submodule_type}_opt_state', None) )

    @typechecked
    def to(self, device_info : TorchDeviceInfo):
        if self._device_info != device_info:
            self._device_info = device_info
            for submodule_type in self.get_submodule_types():
                module = self._sub_modules[submodule_type]
                module.to(device_info.get_device())
                module_opt = self._sub_modules_opt[submodule_type]
                module_opt.load_state_dict(module_opt.state_dict())

    def train(self):
        if self._train_or_eval != True:
            self._train_or_eval = True
            for submodule_type in self.get_submodule_types():
                self._sub_modules[submodule_type].train()

    def eval(self):
        if self._train_or_eval != False:
            self._train_or_eval = False
            for submodule_type in self.get_submodule_types():
                self._sub_modules[submodule_type].eval()
