import threading
from typing import List, Union

import torch
from ..python import dotdict

class TorchDeviceInfo:
    """
    Represents  torch device info
    """
    def __init__(self, backend : str = None,
                       index : int = None,
                       state : Union[dict, None] = None):

        if state is None:
            state = {}

        self._backend = backend or state.get('backend', 'cpu')
        self._index = index or state.get('index', 0)
    
    def state(self) -> dict: return {'backend' : self._backend, 'index': self._index}

    def is_cpu(self) -> bool: return self._backend == 'cpu'

    def get_backend(self) -> str: return self._backend
    def get_index(self) -> int: return self._index
    def get_name(self) -> str: return self._get_real_device().name
    def get_total_memory(self) -> int: return self._get_real_device().total_memory

    def get_device(self) -> torch.device:
        """get physical torch.device"""
        dev = self._get_real_device()
        return torch.device(f'{dev.backend}:{dev.index}' if dev.index is not None else f'{dev.backend}')
    
    def _get_real_device(self):
        # return real device
        for dev in _get_real_devices():
            if dev.backend == self._backend and dev.index == self._index:
                return dev
        # Fallback to CPU
        return _get_cpu_real_device()
        
    def __hash__(self): return (self._backend, self._index).__hash__()
    def __eq__(self, other):
        if self is not None and other is not None and isinstance(self, TorchDeviceInfo) and isinstance(other, TorchDeviceInfo):
            return self._backend == other._backend and self._index == other._index
        return False
    
    def __getstate__(self): return self.state()
    def __setstate__(self, state): self.__init__(state=state)
    
    def __str__(self):
        dev = self._get_real_device()
        if dev.index is not None:
            s = f'[{dev.backend.upper()}:{dev.index}]'
        else:
            s = f'[{dev.backend.upper()}]'
        
        if dev.name is not None:
            s += f' {dev.name}'
        if dev.total_memory is not None:
            s += f' [{(dev.total_memory / 1024**3) :.3}Gb]'
        return s

    def __repr__(self):
        return f'{self.__class__.__name__} object: ' + self.__str__()

_real_devices_lock = threading.Lock()
_real_devices = None

def _get_cpu_real_device(): return dotdict(backend='cpu', index=None, name=None, total_memory=None)

def _get_real_devices():
    global _real_devices_lock
    global _real_devices
    with _real_devices_lock:
        if _real_devices is None:
            _real_devices = [ _get_cpu_real_device() ]
            
           
            for i in range (torch.cuda.device_count()):
                device_props = torch.cuda.get_device_properties(i)
                _real_devices.append(  dotdict(backend='cuda', index=i, name=device_props.name, total_memory=device_props.total_memory) )
            
            if getattr(torch,'dml', None) is not None:
                for i in range (torch.dml.device_count()):
                    _real_devices.append( dotdict(backend='dml', index=i, name=torch.dml.device_name(i), total_memory=0) )
                    
    return _real_devices


def get_cpu_device_info() -> TorchDeviceInfo: return TorchDeviceInfo(backend='cpu', index=0)

def get_avail_devices_info() -> List[TorchDeviceInfo]:
    """returns a list of available TorchDeviceInfo"""
    return [ TorchDeviceInfo(backend=dev.backend, index=dev.index) for dev in _get_real_devices() ]

