from typing import Sequence, Tuple, Union

import numpy as np


class Desc:
    def __init__(self, desc : Union[str, None] = None,
                       state : Union[dict, None] = None,):
        if state is None:
            state = {}
        self._desc = state.get('desc', desc)

    def get_desc(self) -> Union[str, None]: return self._desc

    def get_state(self) -> dict: return {'desc' : self._desc}


class Item(Desc):
    ...

class Image(Item):
    def __init__(self, image : Union[np.ndarray, None] = None,
                       desc : Union[str, None] = None,
                       state : Union[dict, None] = None,
                       ):
        """image   HWC"""
        if image is not None and image.ndim != 3:
            raise ValueError('image.ndim != 3 (HWC)')

        super().__init__(desc=desc, state=state)
        if state is None:
            state = {}

        self._image = state.get('image', image)

    def get_state(self) -> dict:
        d = super().get_state()
        d.update( {'image' : self._image} )
        return d

    def get_image(self) -> np.ndarray: return self._image
    def get_image_shape(self) -> Tuple[int, int, int]: return self._image.shape


_name_by_class = {Image : 'Image',
                 }
_class_by_name = {name : type  for type, name in _name_by_class.items() }

class Section(Desc):
    @typechecked
    def __init__(self, items : Union[Sequence[Item], None] = None,
                       desc : Union[str, None] = None,
                       state : Union[dict, None] = None,
                       ):
        """"""
        super().__init__(desc=desc, state=state)
        if state is None:
            state = {}

        self._items = items or [_class_by_name[ item_state['type_name'] ](state=item_state['state'])
                                for item_state in state.get('items_state', []) ]

    def get_items(self) -> Sequence[Item]: return self._items

    def get_state(self) -> dict:
        d = super().get_state()
        d.update({'items_state' : [ {'type_name' : _name_by_class[item.__class__],
                                     'state' : item.get_state() } for item in self._items ], })
        return d

class Sheet(Desc):
    @typechecked
    def __init__(self, sections : Union[Sequence[Section], None] = None,
                       desc : Union[str, None] = None,
                       state : Union[dict, None] = None ):
                       
        super().__init__(desc=desc, state=state)
        if state is None:
            state = {}

        self._sections = sections or [Section(state=section_state) for section_state in state.get('sections_state', [])]

    def get_sections(self) -> Sequence[Section]:  return self._sections

    def get_state(self) -> dict:
        d = super().get_state()
        d.update({'sections_state' : [section.get_state() for section in self._sections]})
        return d
