from typing import Iterable, Tuple, Union

import cv2
import numpy as np

from ..math import Affine2DMat, Affine2DUniMat


class ImageWarper:#TODO lib_img.aug.Geo

    def __init__(self,
                 align_uni_mat : Union[Affine2DUniMat, None] = None,

                 align_tx : float = 0.0,
                 align_ty : float = 0.0,
                 align_scale : float = 0.0,
                 align_rot_deg : float = 0.0,

                 rw_tx_var : float = 0.2,
                 rw_ty_var : float = 0.2,
                 rw_scale_var : float = 0.3,
                 rw_rot_deg_var : float = 180.0,

                 rt_tx_var : float = 0.30,
                 rt_ty_var : float = 0.30,
                 rt_scale_var : float = 0.40,
                 rt_rot_deg_var : float = 15.0,
                 rt_random_flip : bool = False,

                 rnd_state : Union[np.random.RandomState, None] = None,
                ):
        """
        Max quality one-pass image augmentation using geometric transformations.

            align_uni_mat    optional Affine2DUniMat if you have pre-align mat, for example given from face aligner like umeyama or similar
        """

        align_scale = 1.0 + align_scale if align_scale >= 0 else 1.0 / (1.0 - align_scale )

        rnd_state = np.random.RandomState()
        rnd_state.set_state(rnd_state.get_state()) if rnd_state is not None else ...

        rw_tx = rnd_state.uniform(-rw_tx_var, rw_tx_var)
        rw_ty = rnd_state.uniform(-rw_ty_var, rw_ty_var)
        rw_scale = rnd_state.uniform(1/(1+rw_scale_var), 1+rw_scale_var)
        rw_rot_deg = rnd_state.uniform(-rw_rot_deg_var, rw_rot_deg_var)

        self._rt_tx = rnd_state.uniform(-rt_tx_var, rt_tx_var)
        self._rt_ty = rnd_state.uniform(-rt_ty_var, rt_ty_var)
        
        x = rnd_state.uniform(-1.0, 1.0)*rt_scale_var
        self._rt_scale = 1 / (1 - x) if x < 0 else 1 + x 
        
        self._rt_rot_deg = rnd_state.uniform(-rt_rot_deg_var, rt_rot_deg_var)

        self._rw_grid_cell_count = rnd_state.randint(3,8)

        self._rt_random_flip  = rnd_state.randint(2) == 0 if rt_random_flip else False

        align_uni_mat = align_uni_mat.invert() if align_uni_mat is not None else Affine2DUniMat()

        align_uni_mat = self._align_uni_mat = align_uni_mat * Affine2DUniMat() \
                                                        .translated(-align_tx+0.5, -align_ty+0.5) \
                                                        .scaled(align_scale) \
                                                        .rotated(align_rot_deg) \
                                                        .translated(-0.5,-0.5)

        self._rw_rnd_mat = align_uni_mat * Affine2DUniMat() .translated(0.5, 0.5) \
                                                            .rotated(rw_rot_deg) \
                                                            .translated(-rw_tx, -rw_ty) \
                                                            .scaled(rw_scale) \
                                                            .translated(-0.5,-0.5)

        self._rnd_state_state = rnd_state.get_state()
        self._cached = {}

    def _get_rt_rnd_mat(self, power : float = 1.0)-> Affine2DUniMat:
        return self._align_uni_mat * Affine2DUniMat()   .translated(0.5, 0.5) \
                                                        .rotated( self._rt_rot_deg*power ) \
                                                        .translated(-self._rt_tx*power, -self._rt_ty*power) \
                                                        .scaled( 1.0 + (self._rt_scale - 1.0) * power ) \
                                                        .translated(-0.5,-0.5)


    def get_aligned_random_transform_mat(self, random_transform_power=1.0) -> Affine2DUniMat:
        """
        returns Affine2DUniMat that represents transformation from aligned image to randomly transformed aligned image
        """
        pts = [ [0,0], [1,0], [1,1]]
        src_pts = self._align_uni_mat.invert().transform_points(pts)
        dst_pts = self._get_rt_rnd_mat(power=random_transform_power).invert().transform_points(pts)

        return Affine2DUniMat.from_3_pairs(src_pts, dst_pts)

    def transform(self, img : np.ndarray, out_res : int, random_warp_power : float = 1.0, random_transform_power : float = 1.0) -> np.ndarray:
        """
        transform an image.

        Subsequent calls will output the same result for any img shape and out_res.

            img                 np.ndarray  (HWC)

            out_res             int

            random_warp_power(1.0)   float     0..1  apply random warp

            random_transform(True)  bool    apply random align
        """
        H,W = img.shape[:2]

        key = (H,W, random_warp_power, random_transform_power)
        data = self._cached.get(key, None)
        if data is None:
            rnd_state = np.random.RandomState()
            rnd_state.set_state( self._rnd_state_state )
            self._cached[key] = data = self._gen_remap_grid(H,W, random_warp_power, random_transform_power, out_res, rnd_state=rnd_state )

        remap_grid, mask = data

        new_img = cv2.remap(img, remap_grid, None, interpolation=cv2.INTER_LANCZOS4)
        new_img *= mask

        if self._rt_random_flip:
            new_img = new_img[:,::-1,:]

        return new_img

    def _gen_remap_grid(self, H, W, random_warp_power, random_transform_power, out_res, rnd_state):
        # generate remap grid and mask

        # make identity uniform remap grid
        remap_grid = np.stack(np.meshgrid(np.linspace(0., 1.0, H, dtype=np.float32),
                                          np.linspace(0., 1.0, W, dtype=np.float32)), -1)

        if random_warp_power != 0.0:
            # make random warp diff_grid
            diff_grid = ImageWarper._gen_rw_coord_uni_diff_grid(out_res, self._rw_grid_cell_count, 0.12, rnd_state)

            # transform diff_grid using rw_rnd_mat
            diff_grid = cv2.warpAffine(diff_grid, self._rw_rnd_mat.to_exact_mat(out_res,out_res, W, H), (W,H), flags=cv2.INTER_LINEAR, borderMode=cv2.BORDER_CONSTANT)
            if random_warp_power < 1.0:
                # random_warp_power decreases amount of coords shifting
                diff_grid *= random_warp_power

            # merge diff_grid with remap_grid
            remap_grid += diff_grid

        # scale remap_grid to image size
        remap_grid *= (H-1, W-1)

        # using rt_rnd_mat and expand it to out_res
        img_to_rt_rnd_mat = self._get_rt_rnd_mat(power=random_transform_power).invert().to_exact_mat(W,H,out_res,out_res)

        # warp remap_grid using img_to_rt_rnd_mat to the space of final out_res image
        remap_grid = cv2.warpAffine(remap_grid, img_to_rt_rnd_mat, (out_res,out_res), flags=cv2.INTER_CUBIC, borderMode=cv2.BORDER_REPLICATE )

        # make binary mask to refine image-boundary
        mask = cv2.warpAffine( np.ones( (H,W), dtype=np.uint8), img_to_rt_rnd_mat, (out_res,out_res), flags=cv2.INTER_NEAREST)[...,None]

        return remap_grid, mask

    def _gen_rw_coord_uni_diff_grid(size: int, cell_count, cell_mod, rnd_state) -> np.ndarray:
        """
        generates square uniform random warp coordinate differences

        grid of shape (size, size, 2)  (x,y)

        cell_count(3)        3+

        cell_mod  (0.12)     [ 0 .. 0.24 ]
        """
        cell_count = max(3, cell_count)
        cell_mod = np.clip(cell_mod, 0, 0.24)
        cell_size = 1.0 / (cell_count-1)

        grid = np.zeros( (cell_count,cell_count, 2), dtype=np.float32 )

        grid[1:-1,1:-1, 0:2] += rnd_state.uniform (low=-cell_size*cell_mod, high=cell_size*cell_mod, size=(cell_count-2, cell_count-2, 2) )
        grid = cv2.resize(grid, (size, size), interpolation=cv2.INTER_CUBIC ).astype(np.float32)

        # Linear dump border cells to zero
        border_size = size // cell_count
        dumper = np.linspace(0, 1, border_size, dtype=np.float32)
        grid[:border_size, :,:] *= dumper[:,None,None]
        grid[-border_size:,:,:] *= dumper[::-1,None,None]
        grid[:,:border_size ,:] *= dumper[None,:,None]
        grid[:,-border_size:,:] *= dumper[None,::-1,None]

        return grid
