from typing import Callable, List

import cv2
import numpy as np


class Window:
    def __init__(self, wnd_name, on_hide : Callable = None):
        self._wnd_name = wnd_name
        self._is_showing = False
        self._on_hide = on_hide
        self._img = None

    def call_on_hide(self, on_hide : Callable) -> 'Window':
        self._on_hide = on_hide
        return self
        
    def get_wnd_name(self): return self._wnd_name
        
    def is_showing(self): return self._is_showing
    
    def show(self):
        if not self._is_showing:
            self._is_showing = True            
            cv2.namedWindow(self._wnd_name)
            if self._img is not None:
                self.set_image(self._img)
        
    def set_image(self, img : np.ndarray):        
        self._img = img
        if self._is_showing:
            cv2.imshow(self._wnd_name, img)
        
    def hide(self):
        if self._is_showing:
            self._is_showing = False
            
            if cv2.getWindowProperty(self._wnd_name,cv2.WND_PROP_VISIBLE) == 1:
                cv2.destroyWindow(self._wnd_name)
                
            if self._on_hide is not None:
                self._on_hide()
                
    def destroy(self):
        self.hide()
        self._window_system._on_window_destroy(self)
            
    def process_messages(self):
        if self._is_showing:
            if cv2.getWindowProperty(self._wnd_name,cv2.WND_PROP_VISIBLE) == 0:
                self.hide()
        
class WindowSystem:
    def __init__(self):
        """manage multiple cv windows"""
        self._windows : List[Window] = []
        
    def finalize(self):
        for wnd in self._windows[:]:
            wnd.destroy()
        
    def create_window(self, wnd_name) -> Window:
        if any ( wnd_name == wnd.get_wnd_name() for wnd in self._windows ):
            raise Exception(f'Window with name {wnd_name} already exists.')
        wnd = Window(wnd_name)
        wnd._window_system = self
        
        self._windows.append(wnd)
        return wnd

    def _on_window_destroy(self, wnd : Window):
        self._windows.remove(wnd)

    def process_messages(self):
        cv2.waitKey(1)
        
        for wnd in self._windows[:]:
            wnd.process_messages()
        
